function [u,v,lambda] = SVD_iterative(X, normalization)
% --------------------------------------------------------------------------------------------------
% Determine the first singular vectors and eigenvalue of a matrix using an iterative algorithm that
% can deal with missing data.
% --------------------------------------------------------------------------------------------------
% X             ... nxm matrix
% normalization ... how to normalize factor loadings:
%                   1 ... mean equal to one
%                   2 ... median equal to one
% --------------------------------------------------------------------------------------------------
% sample call: [u,v,lambda] = SVD_iterative([4   1; 1 3])
%              [u,v,lambda] = SVD_iterative([4 1 NaN; 1 3 1; NaN 1 5])
%              [u,v,lambda] = SVD_iterative([1   NaN; NaN NaN])
% --------------------------------------------------------------------------------------------------

% Initialize vectors
[n, m] = size(X);
u = ones(n, 1);
v = ones(m, 1);
u_old = zeros(n, 1);
v_old = zeros(m, 1);

% Determine u,v iteratively
eps = 1e-10;
counter = 0;
while (max(abs(u-u_old)) + max(abs(v-v_old))> eps) & (counter < 20)
    % For given v, determine u via regression
    u_old = u;
    for i=1:n
        % Ignore missing values
        X_i = X(i,:);
        missing_i = isnan(X_i);
        if (sum(missing_i) == m)
            u(i) = 0;
        else
            u(i) = 1/(v(~missing_i)' * v(~missing_i)) * v(~missing_i)' * X_i(~missing_i)';
        end
    end
    
    % For given u, determine v via regression
    v_old = v;
    for j=1:m
        % Ignore missing values
        X_j = X(:,j);
        missing_j = isnan(X_j);
        if (sum(missing_j) == n)
            v(j) = 0;
        else
            v(j) = 1/(u(~missing_j)' * u(~missing_j)) * u(~missing_j)' * X_j(~missing_j);   
        end
    end     
    counter = counter + 1;
end

% Normalize u,v to obtain lambda
if (sum(u) == 0)
    u = NaN(n,1);
    lambda_u = 0;
else
    if (normalization == 1)
        % Average factor loading equal to one
        lambda_u = norm(u) / sqrt(n);
    elseif (normalization == 2)
        % Median factor loading equal to one
        lambda_u = median(u);
    end
    u = u / lambda_u;
end
if (sum(v) == 0)
    v = NaN(m,1);
    lambda_v = 0;
else
    if (normalization == 1)
        % Average factor loading equal to one
        lambda_v = norm(v) / sqrt(m);
    elseif (normalization == 2)
        % Median factor loading equal to one
        lambda_v = median(v);
    end
    v = v / lambda_v;
end
lambda = lambda_u * lambda_v;

% Display matrix approximation
%disp(u * lambda * v');