function TS_new = TS_covariance_TS(TS1, TS2, method_2nd, param_2nd)
% --------------------------------------------------------------------------------------------------
% Calculate the rolling covariance of two (usually return-) time series. The new dates will be the
% union of the dates in TS1 and TS2.
% --------------------------------------------------------------------------------------------------
% TS1           ... first time series
% TS2           ... second time series
% method_2nd    ... method by which to calculate second moments:
%                   1 = empirical second moment of last 'param_2nd' data points
%                   2 = use exponentially weighted moving average with decay factor 'param_2nd', e.g. 0.95
%                   3 = robust estimator based on median of sqrt(abs(values1*value2)) during last 'param_2nd' days
%                   4 = same as 1, but take delta_t into account
%                   5 = estimate of continuous volatility via bi-power variation & taking delta_t into account
% --------------------------------------------------------------------------------------------------
% sample call:
%{
    TS_covariance_TS(TS_return(struct('values', CDX_NA_IG2.portfolio(1).market_price{1}, 'dates', CDX_NA_IG2.portfolio(1).dates{1}), 3), ...
                     TS_return(struct('values', CDX_NA_IG2.portfolio(2).market_price{1}, 'dates', CDX_NA_IG2.portfolio(2).dates{1}), 3), 3, 1, 20)
%}
% --------------------------------------------------------------------------------------------------

% Bring both TS onto same date scale
all_dates = union_sorted_c(TS1.dates, TS2.dates);
TS1 = TS_sample(TS1, all_dates);
TS2 = TS_sample(TS2, all_dates);

% Covariance estimate
if (method_2nd == 1)
    TS_new = TS_moving_average(TS_mult_TS(TS1, TS2), param_2nd);
elseif (method_2nd == 2)
    TS_new = TS_ewma(TS_mult_TS(TS1, TS2), param_2nd);
elseif (method_2nd == 3)
    TS_new = TS_mult_double(TS_rolling_median(TS_mult_TS(TS1, TS2), param_2nd),  1/0.6745);
elseif (method_2nd == 4)
    TS_dt = TS_mult_double(TS_delta_t(TS1), 1/365);
    TS_innovations_std = TS_divide_TS(TS_mult_TS(TS1, TS2), TS_dt);
    TS_new = TS_moving_average(TS_innovations_std, param_2nd);
elseif (method_2nd == 5)
    % Calculate bipower covariation
    TS_bp_cov = TS_bipower_covariation(TS1, TS2);
    % Calculate lagged series and bring on same time scale
    TS_bp_cov_lag = TS_lag(TS_bp_cov, 1, param_2nd);    
    TS_bp_cov_lag = TS_merge(TS_bp_cov_lag, TS_const(0, TS_bp_cov.dates(1)));
    TS_bp_cov_lag = TS_sample(TS_bp_cov_lag, TS_bp_cov.dates);
    % Calculate covariance estimate
    TS_new = TS_minus_TS(TS_bp_cov, TS_bp_cov_lag); 
    TS_new = TS_sub_period(TS_new, TS1.dates(1), TS1.dates(end));
    TS_dt_used = struct('dates', TS_new.dates, 'values', min(param_2nd, daysdif(TS1.dates(1), TS_new.dates)) / 365);
    TS_new = TS_divide_TS(TS_new, TS_dt_used);
end