function log_LH = TS_likelihood(TS, kernel, params)
% --------------------------------------------------------------------------------------------------
% For a given time series and transition kernel, calculate the negative log-likelihood of the
% observed data.
% --------------------------------------------------------------------------------------------------
% TS        ... structure with values and dates
% kernel    ... type of transition kernel
%               1 ... affine jump diffusion AJD(k,theta,sigma,L,mu)
% params    ... structure with parameter time series for transition kernel
% --------------------------------------------------------------------------------------------------
% sample call: TS_likelihood(TS, 1, TS_params);
% --------------------------------------------------------------------------------------------------

% Calculate return time series, delta_t time series & delete last values of original time series
TS_abs_returns = TS_return(TS, 1);
TS_dt = TS_delta_t(TS, 1);
TS_dt = TS_divide_double(TS_dt, 365.24);
period_start_dates = TS.dates(1:end-1);
TS = TS_sample(TS, period_start_dates);

% Cut parameter time series down to same date range as data time series
param_names = fields(params);
TS_params = struct();
for i=1:length(param_names)
    name = param_names{i};
    if (strcmp(name, 'dates'))
        continue;
    end
    TS_params.(name) = TS_sample(struct('values', params.(name), 'dates', params.dates), period_start_dates);
end

% AJD transition kernel
if (kernel == 1)
    % Calculate model-implied expected returns
    TS_decay = TS_exponentiate(TS_mult_double(TS_mult_TS(TS_params.k, TS_dt), -1));
    TS_decay = TS_add_double(TS_mult_double(TS_decay, -1), 1);
    TS_exp_return = TS_mult_TS(TS_minus_TS(TS_params.theta, TS), TS_decay);
    
    % Calculate model-implied volatility of innovationa
    TS_vol = TS_mult_TS(TS_power(TS_mult_TS(TS, TS_dt), 0.5), TS_params.sigma);
    
    % Calculate model-implied time series of jump probabilites
    TS_jump_prob = TS_exponentiate(TS_mult_double(TS_mult_TS(TS_params.L, TS_dt), -1));
    TS_jump_prob = TS_add_double(TS_mult_double(TS_jump_prob, -1), 1);
    
    % Calculate time series of transition likelihoods
    LH_trans = (1-TS_jump_prob.values) .* normpdf(TS_abs_returns.values, TS_exp_return.values, TS_vol.values);
    LH_trans = LH_trans + TS_jump_prob.values .* pdf('exp', TS_abs_returns.values, TS_params.mu.values);
    
    % Sum up individual log-transition-likelioods
    log_LH = zeros(size(LH_trans));
    log_LH(LH_trans > 0) = log(LH_trans(LH_trans > 0));
    log_LH(LH_trans == 0) = -1e10;
    log_LH = -sum(log_LH);
end
