function TS_new = TS_matrix_function(TS_matrix, func, TS_used_rows, TS_used_cols)
% --------------------------------------------------------------------------------------------------
% At each point in time, apply the function 'func' to the vector of currently active time series.
% --------------------------------------------------------------------------------------------------
% TS_matrix         ... cell matrix of time series
% func              ... function to apply, e.g. @mean or @median
% TS_used_rows      ... time series of used rows, default: always use all
% TS_used_cols      ... time series of used columns, default: always use all
% --------------------------------------------------------------------------------------------------
% sample call:
%{
    TS_matrix = {struct('values', [1 5]', 'dates', [732211 732647]')  [];
                 struct('values', [2.5 3.5]', 'dates', [732210 732648]')  []};
    TS_matrix_function(TS_matrix, @mean)
    TS_matrix_function(TS_matrix, @median)
%}
% --------------------------------------------------------------------------------------------------

% Return empty TS if TS_matrix is empty
if isempty(TS_matrix)
    TS_new = [];
    return;
end

% Synchronize time series
TS_matrix = TS_matrix_synchronize(TS_matrix);
num_dates = length(TS_matrix{1}.dates);
num_TS1 = size(TS_matrix, 1);
num_TS2 = size(TS_matrix, 2);
num_ts = num_TS1 * num_TS2;

% By default, use all rows and columns of matrix
if (nargin <= 2)
    TS_tmp = struct('dates', TS_matrix{1,1}.dates, 'values', ones(num_dates, 1));
    TS_used_rows = cell(num_TS1, 1);
    for j=1:num_TS1
        TS_used_rows{j} = TS_tmp;
    end
end
if (nargin <= 3)
    TS_tmp = struct('dates', TS_matrix{1,1}.dates, 'values', ones(num_dates, 1));
    TS_used_cols = cell(num_TS2, 1);
    for j=1:num_TS2
        TS_used_cols{j} = TS_tmp;
    end
end

% Synchronize TS_used_rows and TS_used_cols with TS_matrix
for j=1:num_TS1
    TS_used_rows{j} = TS_sample(TS_used_rows{j}, TS_matrix{1,1}.dates);
end
for j=1:num_TS2
    TS_used_cols{j} = TS_sample(TS_used_cols{j}, TS_matrix{1,1}.dates);
end

% Extract time series values
values_all = NaN(num_dates, num_ts);
for i=1:num_TS1
    for j=1:num_TS2
        if (~isempty(TS_matrix{i,j}))
            % Determine dates on which TS_used_rows(i) and TS_used_cols(j) equal to one
            used_dates = (TS_used_rows{i}.values == 1) & (TS_used_cols{j}.values == 1);
            
            % For these dates, add values to matrix
            values_all(used_dates,i+(j-1)*num_TS1) = TS_matrix{i,j}.values(used_dates);
        end
    end
end

% Apply function 'func' at each point in time
values = zeros(length(TS_matrix{1}.dates), 1);
for i=1:num_dates
    values_t = values_all(i,:);
    missing = isnan(values_t);
    if (sum(missing) == num_ts)
        values(i) = NaN;
    else
        values(i) = func(values_t(~missing));
    end
end

% Return time series with median
TS_new = struct('dates', TS_matrix{1}.dates, 'values', values);
