function vol_ts = TS_price_to_vol(prices, dates, method, param)
% --------------------------------------------------------------------------------------------------
% Convert a price time series into a vol time series object (date, vol, delta_t) using data only up
% to time t-1 for time t estimate.
% --------------------------------------------------------------------------------------------------
% prices    ... price time series
% dates     ... corresponding dates
% method    ... 1 = moving average of length 'param'
%               2 = exponentially weighted moving average with decay factor 'param', e.g. 0.95
%               3 = robust estimator based on medium of squared returns during last 'param' days
% --------------------------------------------------------------------------------------------------
% sample call: TS_price_to_vol(CDX_NA_IG2.portfolio(1).market_price{1}, CDX_NA_IG2.portfolio(1).dates{1}, 1, 40)
%              TS_price_to_vol(CDX_NA_IG2.portfolio(1).market_price{1}, CDX_NA_IG2.portfolio(1).dates{1}, 2, 0.975)
%              TS_price_to_vol(CDX_NA_IG2.portfolio(1).market_price{1}, CDX_NA_IG2.portfolio(1).dates{1}, 3, 40)
% --------------------------------------------------------------------------------------------------

% Convert price to return TS
return_ts = TS_price_to_return(prices, dates);
if length(return_ts.dates) <= 2
    vol_ts = struct('dates', [], 'vol', [], 'delta_t', []);
    return;
end
returns_squared = (return_ts.returns .^ 2) ./ return_ts.delta_t * 365;

% Calculate vol time series - method 1
if method==1
    variance = returns_squared;
    for j = 2:param
        returns_squared = [returns_squared(1); returns_squared(1:end-1);];
        variance = variance + returns_squared;
    end
    variance = variance / param;
    vol = sqrt(variance );
end

% Calculate vol time series - method 2
if method==2
    variance = ewma(returns_squared, param, 0);
    vol = sqrt(variance);
end

% Calculate vol time series - method 3
if method==3
    vol = rolling_median(sqrt(returns_squared), param) / 0.6745;
end

% Return volatility time series object
vol_ts = struct('dates', return_ts.dates, 'vol', vol, 'delta_t', return_ts.delta_t);
