function TS_new = TS_sample(TS, dates2, method)
% --------------------------------------------------------------------------------------------------
% Convert a time  series (prices, dates) into a time series (prices2, dates2), for given 'dates2',
% where 'prices2' is obtained by using the last available date.
% --------------------------------------------------------------------------------------------------
% TS        ... structure with values and dates
% dates     ... desired dates of sampled time series
% method    ... interpolation method:
%               1 ... use last available value (NaN if not possible) - default
%               2 ... same as 1, but don't use NaN - impute first available data point insteads
% --------------------------------------------------------------------------------------------------
% sample call: TS_sample(struct('values', [1 5]', 'dates', [732211 732647]'), [732200 732211 732300 732647 732700]')
% --------------------------------------------------------------------------------------------------

if isempty(TS)
    TS_new = TS;
    return;
end

% By default, use method 1
if (nargin < 3)
    method = 1;
end

% If TS is empty, return NaNs
if isempty(TS.values)
    if isempty(dates2)
        TS_new = [];
    else
        TS_new = struct('dates', dates2, 'values', repmat(NaN, length(dates2), 1));
    end
    return;
end

% Method 1: use last available price
if (method == 1) | (method == 2)
    % Determine number of initial dates2 that are before first time series observation
    all_dates = union_sorted_c(dates2, TS.dates(1));
    num_missing = sum(all_dates <= TS.dates(1))-1;
    
    % For each date in 'dates2' determine date in 'dates' that is smaller or equal
    if (0)
        % Matlab implementation
        pos = zeros(length(dates2) - num_missing, 1);
        for i=(num_missing+1):length(dates2)
            tmp = find(TS.dates <= dates2(i), 1, 'last');
            pos(i-num_missing) = tmp;
        end
    else
        % C implementation
        used_dates2 = dates2((num_missing+1):end);
        if isempty(used_dates2)
            pos = [];
        else
            pos = previous_pos_c(TS.dates, used_dates2);
        end
    end
    if (method == 1)
        values2 = [NaN(num_missing,1); TS.values(pos)];
    elseif (method == 2)
        values2 = [repmat(TS.values(pos(1)), num_missing, 1); TS.values(pos)];
    end
end

% Return new time series
TS_new = struct('dates', dates2, 'values', values2);
