function TS_new = TS_volatility(TS, method, param)
% --------------------------------------------------------------------------------------------------
% Convert a (usually return-) time series into a volatility time series.
% --------------------------------------------------------------------------------------------------
% TS        ... structure with values and dates
% method    ... 1 = moving average of length 'param'
%               2 = exponentially weighted moving average with decay factor 'param', e.g. 0.95
%               3 = robust estimator based on median of absolute returns during last 'param' days
%               4 = same as 1, but take delta_t into account
%               5 = estimate of continuous volatility via bi-power variation & taking delta_t into account
% --------------------------------------------------------------------------------------------------
% sample call:
%{
    test1=TS_volatility(TS_return(struct('values', CDX_NA_IG2.portfolio(1).market_price{1}, 'dates', CDX_NA_IG2.portfolio(1).dates{1}), 3), 1, 40)
    test2=TS_volatility(TS_return(struct('values', CDX_NA_IG2.portfolio(1).market_price{1}, 'dates', CDX_NA_IG2.portfolio(1).dates{1}), 3), 2, 0.975)
    test3=TS_volatility(TS_return(struct('values', CDX_NA_IG2.portfolio(1).market_price{1}, 'dates', CDX_NA_IG2.portfolio(1).dates{1}), 3), 3, 40)
%}
% --------------------------------------------------------------------------------------------------

if (isempty(TS)) | (length(TS.dates) == 1)
    TS_new = [];
    return;
end

% Volatility estimate based on moving average of squared returns
if (method == 1)
    TS_squared = TS_power(TS, 2);
    TS_var = TS_moving_average(TS_squared, param);
    TS_new = TS_function(TS_var, @sqrt);
elseif (method == 2)
    TS_squared = TS_power(TS, 2);
    TS_var = TS_ewma(TS_squared, param);
    TS_new = TS_function(TS_var, @sqrt);
elseif (method == 3)
    TS_abs = TS_function(TS, @abs);
    TS_new = TS_mult_double(TS_rolling_median(TS_abs, param),  1/0.6745);
elseif (method == 4)
    TS_squared = TS_power(TS, 2);
    TS_dt = TS_mult_double(TS_delta_t(TS), 1/365);
    TS_innovations_std = TS_divide_TS(TS_squared, TS_dt);
    TS_var = TS_moving_average(TS_innovations_std, param);
    TS_new = TS_function(TS_var, @sqrt);    
elseif (method == 5)
    % Calculate bipower variation
    TS_bp = TS_bipower_variation(TS);
    % Calculate lagged series and bring on same time scale
    TS_bp_lag = TS_lag(TS_bp, 1, param);    
    TS_bp_lag = TS_merge(TS_bp_lag, TS_const(0, TS_bp.dates(1)));
    TS_bp_lag = TS_sample(TS_bp_lag, TS_bp.dates);
    % Calculate volatility estimate
    TS_new = TS_minus_TS(TS_bp, TS_bp_lag); 
    TS_new = TS_sub_period(TS_new, TS.dates(1), TS.dates(end));
    TS_dt_used = struct('dates', TS_new.dates, 'values', min(param, daysdif(TS.dates(1), TS_new.dates)) / 365);
    TS_new = TS_divide_TS(TS_new, TS_dt_used);
    TS_new = TS_function(TS_new, @sqrt); 
end
