function interpolated = cubic_interpolation_matlab_c(x, y, dz, num_points)
% --------------------------------------------------------------------------------------------------
% Hybrid Matlab/C implementation of cubic spline interpolation in order to achieve better speed.
% --------------------------------------------------------------------------------------------------
% x         ... x-coordinates where function is known
% y         ... function values at x-coordinates
% dz        ... spacing of grid for which interpolation is desired
% N         ... length of desired grid: 0, dz, 2*dz, ..., (N-1)*dz
% --------------------------------------------------------------------------------------------------
% sample call: cubic_interpolation_matlab_c((0:5), [1 1.5 2 2 1.5 1.4], (0:0.1:5))
% --------------------------------------------------------------------------------------------------

%interpolated = spline(x, y, xx);

% Set up the sparse, tridiagonal, linear system b = ?*c for the slopes
n = length(x);
dx = diff(x);
divdif = diff(y) ./ dx;
b = zeros(1,n);
b(1,2:n-1) = 3*(dx(1,2:n-1) .* divdif(1,1:n-2) + dx(1,1:n-2) .* divdif(1,2:n-1));
% Calculate end slopes
x31 = x(3)-x(1);
xn = x(n)-x(n-2);
b(1,1) = ((dx(1)+2*x31) * dx(2) * divdif(1,1) + dx(1)^2 * divdif(1,2)) /x31;
b(1,n) = (dx(n-1)^2 * divdif(1,n-2) + (2*xn+dx(n-1)) * dx(n-2) * divdif(1,n-1)) / xn;
% Sparse linear equation solution for the slopes
dxt = dx(:);
c = spdiags([ [x31;     dxt(1:n-2);                 0] ...
              [dxt(2);  2*(dxt(2:n-1)+dxt(1:n-2));  dxt(n-2)] ...
              [0;       dxt(2:n-1);                 xn] ], [-1 0 1], n, n);
s = b / c;

% Define cubic spline
pp = pwch(x, y, s, dx, divdif);
pp.dim = 1;

%%%%%%%%%%%%%%%%%%%%%%%%%
% Evaluate cubic spline %
%%%%%%%%%%%%%%%%%%%%%%%%%
%interpolated = ppval(pp, xx);

% Take apart PP
[b,c,l,k] = unmkpp(pp);

if (0)      % Matlab implementation
    % For each evaluation site, compute its breakpoint interval
    %[ignored,index] = histc(xx, [-inf,b(2:l), inf]);  % Replace by histc_sorted_c
    xx = (0:(N-1)) * dz;
    index = double(histc_sorted_c(xx, [-inf,b(2:l)]));
    %index2 = double(histc_sorted_c2(dz, N, [-inf,b(2:l)]));

    % Go to local coordinates
    x2 = xx-b(index);

    % Apply nested multiplication:
    interpolated = c(index,1);
    for i=2:k
       interpolated = x2(:) .* interpolated + c(index,i);
    end
    interpolated = interpolated';
else        % C implementation
    %interpolated = evaluate_cubic_spline_c(xx, b, c);
    interpolated = evaluate_cubic_spline_c2(dz, num_points, b, c);
end

