/*=================================================================
 *
 * empirical_quantiles.c ...  Calculate empirical quantiles for given CDF. Return positions matching
 *                            positions in cumulative density
 *
 * The calling syntax is:
 *
 *		pos = empirical_quantiles(cum_density, quantiles)
 *
 * sample: empirical_quantiles(sqrt(0:0.1:1)', [0.3; 0.5; 0.7; 0.9])
 *
 *=================================================================*/

#include <math.h>
#include "mex.h"

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/* Recursive computation of loss distribution */
static void empirical_quantiles(double cum_density[], double quantiles[], double pos[], double num_elements1, double num_elements2)
{
    int i, j;
    double tmp;
    
    /* For each horizon and each data, calculate survival probability */
    j = 0;
    for (i=0; i<num_elements2; i++)
    {
        tmp = quantiles[i];
        while ((j < num_elements1) & (cum_density[j] < tmp - 1e-10))
        {
            j++;
        }
        tmp = (double) j;
        pos[i] = max(1, tmp + 0.5);
    }
    return;
}


/* Gateway routine (to Matlab) */
void mexFunction( int nlhs, mxArray *plhs[], 
		  int nrhs, const mxArray*prhs[] )
     
{ 
    double *cum_density, *quantiles;
    double *pos, M, N;
    unsigned int num_elements1, num_elements2; 

    /* Get number of elements in both sets */
    M = mxGetM(prhs[0]); 
    N = mxGetN(prhs[0]);
    num_elements1 = max(M, N);
    if ((M==0) | (N==0))
    {
        num_elements1 = 0;
    }
    M = mxGetM(prhs[1]); 
    N = mxGetN(prhs[1]);
    num_elements2 = max(M,N);
    if ((M==0) | (N==0))
    {
        num_elements2 = 0;
    }
    
    /* Assign pointers to the input variables parameters */ 
    cum_density = mxGetPr(prhs[0]);
    quantiles = mxGetPr(prhs[1]);
    
    /* Create a mtrix for the return arguments */ 
    plhs[0] = mxCreateDoubleMatrix(num_elements2, 1, mxREAL);
    pos = mxGetPr(plhs[0]);
        
    /* Do the actual computations in a subroutine */
    if (num_elements2 > 0)
    {
        empirical_quantiles(cum_density, quantiles, pos, num_elements1, num_elements2);
    }
    /* p[0] = start[0]; */
    /* p[1] = start[1]; */
    return;
}
