/*=================================================================
 *
 * evaluate_cubic_spline_c2.c ... Similar to ppval in Matlab, but faster since optimized for cubic splines AND equally spaced data
 *
 * The calling syntax is:
 *
 *		evaluate_cubic_spline_c.c(xx, b, c, index)
 *
 *  dz      ... spacing of points where to evaluate cubic spline
 *  N       ... number of observations: 0, dz, 2*dz, ..., (N-1)*dz
 *  b       ... support points of spline
 *  c       ... spline coefficients
 *  index   ... matching coefficient rows for points xx
 *
 * Sample call:
 *
 *      evaluate_cubic_spline_c2(0:0.1:1, [0.2 0.8], [1 1 1 1; 2 2 2 2])
 *
 *=================================================================*/

#include <math.h>
#include "mex.h"

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/* Compute histogram bins */
static void histc_sorted_c2(int bins[], double dz[], double edges[], int num_points_int, unsigned int num_edges)
{
    int i, j;
    
    i = j = 0;
    while (i < num_points_int)
    {
        while ((dz[0] * i > edges[j] - 3e-11) && (j < num_edges))
        {
        	j++;
        }
        bins[i] = j;
        i++;
    }
    
    return;
}

/* Evaluate cubic spline at xx */
static void evaluate_cubic_spline_c2(double dz[], double yy[], double b[], double c[], int index[], int num_points_int, unsigned int num_breaks)
{
    int i, k, c_length, tmp, tmp2;
    double xx_local2;
    
    /* Initialize y */
    /*for (i = 0; i<num_points_int; i++) {
        yy[i] = c[index[i]-1];
    }*/
    
    /* Apply nested multiplication */
    c_length = num_breaks - 1;
    for (k = 1; k<4; k++) {        
        tmp = c_length * k - 1;
        for (i = 0; i<num_points_int; i++) {
            /* Go to local coordinates */
            tmp2 = index[i]-1;
            xx_local2 = dz[0]*i - b[tmp2];
            if (k==1) {                                
                yy[i] = c[tmp2];
            }
            
            /* Evaluate spline */
            yy[i] = xx_local2 * yy[i] + c[index[i] + tmp];
        }
    }

    return;
}


/* Gateway routine (to Matlab) */
void mexFunction( int nlhs, mxArray *plhs[], 
		  int nrhs, const mxArray*prhs[] )
     
{ 
    int i;
    double *xx, *yy, *b, *c, tmp_b0, *dz, *num_points; 
    unsigned int num_points1, num_points2, num_breaks;
    int num_points_int;
    int *index;
    const int *dims;
    
    /* Get number of data points */
    dz = mxGetPr(prhs[0]);
    num_points = mxGetPr(prhs[1]);
    num_points_int = (int) num_points[0];
    
    num_points1 = mxGetM(prhs[2]);
    num_points2 = mxGetN(prhs[2]);
    num_breaks = max(num_points1, num_points2);
    
    /* Allocate memory */
    plhs[0] = mxCreateDoubleMatrix(1, num_points_int, mxREAL);
    yy = mxGetPr(plhs[0]);
    if (true) {
        /* Allocate memory, faster than Matlab version below */
        index = malloc (sizeof (i) * num_points_int);
    } else {
        /*xx_local =  mxGetPr(mxCreateDoubleMatrix(num_points_int, 1, mxREAL));
        dims = mxGetDimensions(prhs[0]);
        index = mxGetPr(mxCreateNumericArray(2, dims, mxINT32_CLASS, 0)); */
    }
    
    /* Assign pointers to the various parameters */ 
    b = mxGetPr(prhs[2]);
    c = mxGetPr(prhs[3]);
    
    /* For each evaluation site, compute its breakpoint interval */
    tmp_b0 = b[0];
    b[0] = -1e20;
    histc_sorted_c2(index, dz, b, num_points_int, num_breaks);
    b[0] = tmp_b0;
    
    /* Go to local coordinates and apply nested multiplication */
    evaluate_cubic_spline_c2(dz, yy, b, c, index, num_points_int, num_breaks);
    
    /* Free up allocated memory */
    if (true) {
        free(index);
    }
    return;
}
