/*=================================================================
 *
 * histc_sorted_c2.c ... Same as Matlab's 'histc', but for sorted AND equally spaced data => faster
 *
 * The calling syntax is:
 *
 *		histc_sorted_c(dz, N, edges)
 *
 *  dz      ... spacing of observations
 *  N       ... number of observations: 0, dz, 2*dz, ..., (N-1)*dz
 *  edges   ... bin boundaries
 *
 * Sample call:
 *
 *      histc_sorted_c2(3, 3, [2 8])        ... same as:    histc_sorted_c([0 3 6], [2 8])
 *      histc_sorted_c2(2.5, 2, [2 7.5])    ... same as:    histc_sorted_c([0 2.5], [2 8])
 *
 *=================================================================*/

#include <math.h>
#include "mex.h"

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/* Compute histogram bins */
static void histc_sorted_c2(int bins[], double dz[], double edges[], int num_points_int, unsigned int num_edges)
{
    int i, j;
    
    i = j = 0;
    while (i < num_points_int)
    {
        while ((dz[0] * i > edges[j] - 3e-11) && (j < num_edges))
        {
        	j++;
        }
        bins[i] = j;
        i++;
    }
    
    return;
}


/* Gateway routine (to Matlab) */
void mexFunction( int nlhs, mxArray *plhs[], 
		  int nrhs, const mxArray*prhs[] )
     
{ 
    double *x, *edges, *dz, *num_points; 
    unsigned int num_points1, num_points2, num_edges;
    int num_points_int;
    int *bins;
    int *dims;
    
    /* Get number of data points */
    dz = mxGetPr(prhs[0]);
    num_points = mxGetPr(prhs[1]);
    num_points_int = (int) num_points[0];
    num_points1 = mxGetM(prhs[2]);
    num_points2 = mxGetN(prhs[2]);
    num_edges = max(num_points2, num_points2);
    
    /* Allocate memory */
    dims = malloc(2 * sizeof(int));
    dims[0] = num_points_int;
    dims[1] = 1;
    plhs[0] = mxCreateNumericArray(1, dims, mxINT32_CLASS, 0);
    bins = mxGetPr(plhs[0]);
    
    /* Assign pointers to the various parameters */ 
    edges = mxGetPr(prhs[2]);
    
    /* Do the actual computations in a subroutine */
    histc_sorted_c2(bins, dz, edges, num_points_int, num_edges);
    return;
}
