function zi = sparse_density(density, dz, num_quantiles, min_support, max_support)
% --------------------------------------------------------------------------------------------------
% Convert a density with regular spacings into a sparse representation based on quantiles. The mean
% and variance are kept the same.
% --------------------------------------------------------------------------------------------------
% density           ... density at points dz*(j-1)
% dz                ... spacing of density
% num_quantiles     ... number of points for new density approximation
% min_support       ... force points to be >= min_support
% max_support       ... force points to be <= max_support
% --------------------------------------------------------------------------------------------------
% sample call: sparse_density([1:100; 50:(-0.5):0.5], [1; 1], 10)
%              sparse_density([1:100], 1, 10)
% --------------------------------------------------------------------------------------------------

% Allocate memory
num_dates = size(density, 1);
n = size(density, 2);
zi = zeros(num_dates, num_quantiles);
grid_quantiles = (1/2/num_quantiles):(1/num_quantiles):(1-1/2/num_quantiles);

for j=1:num_dates
    % Determine support points for density
    dens = density(j,:) / sum(density(j,:));
    cum_density = cumsum(dens);

    if (0)
        % Old method
        for i=1:num_quantiles
            % Extract point position
            pos = find(cum_density >= grid_quantiles(i), 1, 'first');
            zi(j,i) = (pos + max(pos-1,1))/2; %*dz(j);
        end
    else
        % Faster method in C
        zi(j,:) = empirical_quantiles(cum_density, grid_quantiles)*dz(j);
    end
        
    % Add correction so that old and new mean are the same
    mean_old = sum(dens .* (0:(n-1))) * dz(j);
	mean_new = sum(zi(j,:) / num_quantiles);
    zi(j,:) = zi(j,:) - (mean_new-mean_old);
    
    % Correction so that old and new variance are the same
    variance_old = sum(dens .* (0:(n-1)).^2) * dz(j).^2 - mean_old^2;
    variance_new = sum(zi(j,:).^2) / num_quantiles - mean_old^2;
    zi(j,:) = (zi(j,:)-mean_old) * sqrt(variance_old/variance_new) + mean_old;
end

% Enforece support boundaries
if (nargin > 3)
    zi = max(min_support, zi);
end
if (nargin > 4)
    zi = min(max_support, zi);
end
