function [quantiles, weights, weights_middle] = sparse_density3(density, dz, num_quantiles, k)
% --------------------------------------------------------------------------------------------------
% Convert a density with regular spacings into a sparse representation based on (i) quantiles and
% (ii) the value of the density at these quantiles (iii) the value of the density at k points
% inbetween each adjacent pair of quantiles where the position of the points corresponds to the
% support points of the Gauss-Legendre numerical quadrature.
% --------------------------------------------------------------------------------------------------
% density           ... density at points dz*(j-1)
% dz                ... spacing of density
% num_quantiles     ... number of points for new density approximation
% k                 ... number of Gauus-Legendre points points between each adjacent pair of quantile points
% --------------------------------------------------------------------------------------------------
% sample call: [points, weights, weights_middle] = sparse_density3(1:100, 1, 10, 2)
% --------------------------------------------------------------------------------------------------

% Determine support points for density
num_quantiles = num_quantiles - 2;
cum_density = cumsum(density);
cum_density = cum_density / cum_density(end);
points_pos = [];
K = num_quantiles;
counter = 0;
while (length(points_pos) < K * 0.5) & (counter < 6)
    % Define quantiles and invert CDF
    counter = counter + 1;
    grid_quantiles = [0 (1/2/num_quantiles):(1/num_quantiles):(1-1/2/num_quantiles) 1];
    tmp = empirical_quantiles(cum_density, grid_quantiles);
    
    % Force quantile to be apart lesst than maximum/K*50
    max_diff = tmp(end)/K*50;
    diff_large = find(diff(tmp) > max_diff);
    tmp_new = tmp;
    for j = length(diff_large):(-1):1
        pos = diff_large(j);
        tmp_new = [tmp(1:(pos-1)); (tmp(pos):max_diff:tmp(pos+1))'; tmp_new(pos+1:end)];
    end
    tmp = tmp_new;
    
    % Determine point position and density
    [points_pos, index] = unique(round(tmp));
    num_quantiles = num_quantiles * 2;
end

% Define position of Gauss-Legendre points
w = {[0]};
w{2} = [-1/sqrt(3) 1/sqrt(3)];
w{3} = [-sqrt(15)/5 0 sqrt(15)/5];
w{4} = [-1/35*sqrt(525+70*sqrt(30))  -1/35*sqrt(525-70*sqrt(30))  1/35*sqrt(525-70*sqrt(30))  1/35*sqrt(525+70*sqrt(30))];
w{5} = [-1/21*sqrt(245+14*sqrt(70))  -1/21*sqrt(245-14*sqrt(70))  0  1/21*sqrt(245-14*sqrt(70))  1/21*sqrt(245+14*sqrt(70))];

% Determine position of k equally spaced points inbetween each pair of adjacent quantile points
points_pos_middle = zeros(k, length(index)-1);
for j=1:k
    ww = 1 - (1 + w{k}(j)) / 2;
    points_pos_middle(j,:) = ww * tmp(index(1:end-1)) + (1-ww) * tmp(index(2:end));
end

% Determine corresponding weightes
quantiles = points_pos' * dz;
weights = density(points_pos);  

% Use linear interpolation to calculate density in middle of quantiles
ww = (points_pos_middle - floor(points_pos_middle));
weights_middle = density(floor(points_pos_middle)) .* (1-ww) + density(ceil(points_pos_middle)) .* ww;  
