/*=================================================================
 *
 * union_sorted_c.c ...  take the union of two sorted arrays of numbers
 *
 * The calling syntax is:
 *
 *		union_sorted_c(x, x2)
 *
 * x, x2    ... sorted array of numbers
 *
 * Sample call:
 *  
 *      union_sorted_c([2 3 4 5 6], [2.5 3 3.5])
 *=================================================================*/

#include <math.h>
#include "mex.h"

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/* Recursive computation of loss distribution */
static void union_sorted_c(double merged[], unsigned int *num_points_merged, double x[], double x2[], unsigned int num_points_x, unsigned int num_points_x2)
{
    int i, j; 
    
    /* Merge elements until ent of array 'x' is reached */
    num_points_merged[0] = i = j = 0;
    while ((i < num_points_x) && (j < num_points_x2))
    {
        /* Check if x[i] >, <, or == x2[j] */
        if (x[i] < x2[j] + 1e-10) {           
            if (x2[j] < x[i] + 1e-10) {     /* equality */
                merged[num_points_merged[0]] = x[i];
                i++;
                j++;
            } else {                        /* x2[j] > x[i] */
                merged[num_points_merged[0]] = x[i];
                i++;
            }
        } else {    /* x[i] > x2[j] */
            merged[num_points_merged[0]] = x2[j];
            j++;
        }
        num_points_merged[0]++;
    }
    
   /* Merge elements from 'x' if any remaining */
    while (i < num_points_x)
    {
        merged[num_points_merged[0]] = x[i++];
        num_points_merged[0]++;
    }
    
    /* Merge elements from 'x2' if any remaining */
    while (j < num_points_x2)
    {
        merged[num_points_merged[0]] = x2[j++];
        num_points_merged[0]++;
    }
    return;
}


/* Gateway routine (to Matlab) */
void mexFunction( int nlhs, mxArray *plhs[], 
		  int nrhs, const mxArray*prhs[] )
     
{ 
    int i;
    double tmp, *x, *x2, *merged, *output; 
    unsigned int tmp2, num_points1, num_points2, num_points_x, num_points_x2, *num_points_merged; 

    /* Get number of data points */
    num_points1 = mxGetM(prhs[0]);
    num_points2 = mxGetN(prhs[0]);
    num_points_x = max(num_points1, num_points2);
    num_points1 = mxGetM(prhs[1]);
    num_points2 = mxGetN(prhs[1]);
    num_points_x2 = max(num_points1, num_points2);
    
    /* Allocate memory, faster than Matlab version below */
    tmp = 0;
    merged = malloc (sizeof (tmp) * (num_points_x + num_points_x2));    
    tmp2 = 0;
    num_points_merged = malloc (sizeof (tmp2));
    
    /* Assign pointers to the various parameters */ 
    x = mxGetPr(prhs[0]);
    x2 = mxGetPr(prhs[1]);
        
    /* Do the actual computations in a subroutine */
    union_sorted_c(merged, num_points_merged, x, x2, num_points_x, num_points_x2); 
    
    /* Cut down array to number of unique elements */
    plhs[0] = mxCreateDoubleMatrix(num_points_merged[0], 1, mxREAL);
    output = mxGetPr(plhs[0]);
    for (i = 0; i < num_points_merged[0]; i++) {
        output[i] = merged[i];
    }
    
    /* Free up memory */
    free(merged);
    free(num_points_merged);
    return;
}
