%--------------------------------------------------------------------------------------------------
% PURPOSE:  Demonstrate low-level functions of tranche pricing in the basic AJD framework:
%       1.) Evaluate moment-generating function of basic AJD
%       2.) Evaluate characteristic function of basic AJD
%       3.) Calculate of density of integrated basic AJD
%       4.) Modified ASB algorithm for calculating condition portfolio loss distribution
%       5.) Calculate the distribution of the number of defaults in the portfolio
%---------------------------------------------------------------------------------------------------
% USAGE: Just type 'demo2' on the Matlab command prompt and press <ENTER>
%---------------------------------------------------------------------------------------------------
% REFERENCES: "Computational Techniques for basic Affine Models of Portfolio Credit Risk", Andreas Eckner, August 2007
%             "Risk Premia in Structured Credit Derivatives", Andreas Eckner, August, 2007.
%---------------------------------------------------------------------------------------------------
% Written by Andreas Eckner
%---------------------------------------------------------------------------------------------------


disp('***************************************************************')
disp('* Demonstrate call of moment-generating function of basic AJD *')
disp('***************************************************************')
disp(' ')

% NOTE: C-files need to be pre-complied
if (0)
    mex mgf_intAJD_c.c
end

% Calling syntax: mgf_intAJD_c(x0, k, theta, sigma, L, mu, T, q)
%  x0       ... initial value of basic AJD
%  k        ... mean reversion speed
%  theta    ... mean reversion level
%  sigma    ... volatility
%  L        ... jump intensity
%  mu       ... expected jump size of exponentially distributed jumps
%  T        ... maximum time horizon
%  q        ... value at which to evaluate moment generating function (-1 for survival probabilities)  
mgfs = mgf_intAJD_c(0.01, 0.25, 0.02, 0.05, 0.02, 0.03, 5, -1)

% This also works (allows user-defined time horizons):
if (0)
    mex mgf_intAJD2_c.c
end
% mgf_intAJD2_c([0.01; 0.01], [0.1 0.25], [0.02; 0.02], [0.05; 0.05], [0.02; 0.02], [0.03; 0.03], [-1; -1], [1 5; 2 5])

h = msgbox({'Evaluated moment generating function at q=-1, with time horizons 0.25, 0.5, 0.75, ..., 4.75, 5 years.' ...
            '' ...
            'The basic AJD parameters are x0=0.01, k=0.25, theta=0.02, L=0.05, mu=0.02.' ...
            '' ...
            'Press OK to continue.'}, 'Step 1', 'none');
uiwait(h);


disp('************************************************************')
disp('* Demonstrate call of characteristic function of basic AJD *')
disp('************************************************************')
disp(' ')

% NOTE: C-files need to be pre-complied
if (0)
    mex transform2_integrated_ajd_c.c
end

[a,b] = transform2_integrated_ajd_c(zeros(10,1), (1:10)', 0.01, 0.25, 0.02, 0.05, 0.02, 0.03, 5);
complex(a,b)

h = msgbox({'Evaluated charateristic function at u=1, 2, ..., 10 with time horizon 5 years.' ...
            '' ...
            'The basic AJD parameters are x0=0.01, k=0.25, theta=0.02, L=0.05, mu=0.02.' ...
            '' ...
            'Press OK to continue.'}, 'Step 2', 'none');
uiwait(h);


disp('**************************************************************')
disp('* Demonstrate calculation of density of integrated basic AJD *')
disp('**************************************************************')
disp(' ')

% Calculate density
N = 1000;
[dens_int_bAJD, dz] = density_integrated_ajd5(0.01, 0.25, 0.02, 0.05, 0.02, 0.03, 5, N);

% Make nice plot
grid = dz*(0:(N-1));
plot(grid, dens_int_bAJD)
xlabel('Integrated basic AJD')
ylabel('Density')
axis([0 1 0 ceil(max(dens_int_bAJD))])

h = msgbox({'Calculated density of integrated basic AJD.' ...
            '' 'The basic AJD parameters are x0=0.01, k=0.25, theta=0.02, L=0.05, mu=0.02.' ...
            '' 'Press OK to continue.'}, 'Step 3', 'none');
uiwait(h);


disp('**************************************')
disp('* Demonstrate modified ASB algorithm *')
disp('**************************************')
disp(' ')

% NOTE: C-files need to be pre-complied
if (0)
    mex cond_loss_dist_c2.c
end

ai = ones(1,10);                        % Loadings on common factor
surv_prop_idio = repmat(0.9, 1, 10);    % Survival probabiliteis for common factor identical zero 
int_common_factor = 0.1;                % Integrated common factor

% Important Note: This function is for demonstration purposes only, since the overhead when calling a
% mex-file makes the use of this function unfeasible. The preferred function is
% 'uncond_loss_dist3.c', which combines the mASB algorithm and the numerical integration over the
% conditional portfolio loss distribution.
cond_loss_dist_c2(ai, surv_prop_idio, int_common_factor)

h = msgbox({'Using the modified ASB algorithm, calculated distribution of number of defaults in portfolio conditional on the integrated common factor being equal to 0.1.' ...
            '' 'The portfolio AJD parameters are m=10 companies and idiosyncratic survival probabilities equal to 0.9.' ...
            '' 'Press OK to continue.'}, 'Step 4', 'none');
uiwait(h);


disp('************************************************************************************************')
disp('* Demonstrate calculating the unconditional portfolio loss distribution at 5-year time horizon *')
disp('************************************************************************************************')
disp(' ')

% Create artifical CDS, CDX index and CDX tranche dataset
if (~exist('cdx_demo'))
    create_cdx_demo2;
end
% Create 3-dim array. Entry (i,j,k) is the probability of seeing (k-1) defaults at date i, up to
% time horizon j.
def_dist = port_loss_dist3(datenum('12/05/2005'), cdx_demo.T{1}, cdx_demo.portfolio, cdx_demo.index_members, cdx_demo.y0, cdx_demo.AJD_common_factor, 2^19);

% Plot distribution up to 30 defaults
bar(0:30, squeeze(def_dist(1,1,1:31)))
axis([-1 31, 0, 0.25])
xlabel('Number of Defaults')
ylabel('Probability')

h = msgbox({'Demonstrated calculating the unconditional portfolio loss distribution for the 5-year index on December 5, 2005, which has a maturity of 5.04 years.' ...
            '' 'The joint default intensity model is contained in the structure ''cdx_demo''.' ...
            '' 'Press OK to continue.'}, 'Step 5', 'none');
uiwait(h);
