%--------------------------------------------------------------------------------------------------
% PURPOSE:  Demonstrate mid-level functions of tranche pricing in the basic AJD framework:
%       1.) Parameter optimization
%       2.) Load fitted parameters
%       3.) Save fitted parameters       
%---------------------------------------------------------------------------------------------------
% USAGE: Just type 'demo3' on the Matlab command prompt and press <ENTER>
%---------------------------------------------------------------------------------------------------
% REFERENCES: "Computational Techniques for basic Affine Models of Portfolio Credit Risk", Andreas Eckner, August 2007
%             "Risk Premia in Structured Credit Derivatives", Andreas Eckner, August, 2007.
%---------------------------------------------------------------------------------------------------
% Written by Andreas Eckner
%---------------------------------------------------------------------------------------------------


disp('*****************************************************************')
disp('* Demonstrate parameter optimization for single date (12/5/2005)*')
disp('*****************************************************************')
disp(' ')

% Create artifical CDS, CDX index and CDX tranche dataset
if (~exist('cdx_demo'))
    create_cdx_demo2;
end

button = questdlg('Do you want to run one iteration of the parameter optimization ? (takes about 2-10 minutes, depending on processor speed)', ...
                      'Step 3', 'Yes', 'No', 'No');

if strcmp(button,'Yes')
    disp('* Determine original RMSE pricing error.*')
    [trash, cdx_demo] = wrapper_tranche_mispricing(get_x0(cdx_demo, [datenum('12/05/2005')]), cdx_demo, discount_curves_demo, datenum('12/05/2005'), ...
                                           datenum('12/05/2005'), 2^18, [1 0 0 1], [1 0 0], 121, 0, 2, 'mu*L', 'thet*k');
    % REMARK: Use this call to determine RMSE for both dates
%     [trash, cdx_demo] = wrapper_tranche_mispricing(get_x0(cdx_demo, [datenum('12/05/2005'); datenum('01/05/2006')]), cdx_demo, ...
%                         discount_curves_demo, datenum('12/05/2005'), datenum('1/05/2006'), 2^18, [1 0 0 1], [1 0 0], 121, 0, 2, 'mu*L', 'thet*k');
    rmse_total_orig = cdx_demo.rmse(1);
    
    disp('Optimizing parameters to fit 5-year tranche spreads on 12/05/2005')
    disp('Later iterations are much faster, since the search interval is shrunk adaptively based on change of parameter value in last intertion.')
    disp('Suggestion: Use Nelder-Mead instead of iterative search algorithm.')
    disp(' ')

    % Do one iteration of parameter optimization - the parameters of the function call are:
        % cdx                           ... CDX index and tranche structure object
        % discount_curves               ... discount curves matching coupon and half-way in-between coupon payment dates
        % start_date_num, end_date_num  ... date-range for which to fit parameters
        % cds_maturities                ... indicator vector, which subset of {5,7,10,1}-maturities to fit for CDS
        % CDX_maturities                ... indicator vector, which subset of {5,7,10,1}-maturities to fit for tranches
        % RMSE_method                   ... which definition of RMSE pricing error to use, this is a
        %                                   3-digit number, determining index, cds, and tranche RMSE definition (see
        %                                   'wrapper_tranche_mispricing' for details
        % LGD_method                    ... method for (joint) distribution of LGDs (0=deterministic,...), see 'value_fixed_leg_tranches' and ...
        %                                   'value_default_leg_tranches' for details
        % used_params                   ... indicator vector of length 10, which parameters to update
        %                                   [k,theta or theta*k,sigma,l,mu or mu*L,omega1,omega2,y0,liq_prem_cds,liq_prem_tranches]
        % method5                       ... which definition to use for the 5th parameter
        %                                   'mu' = expected jump size \mu (default)
        %                                   'mu*L' = jump intensity x jump size
        % method2                       ... which definition to use for the 2nd parameter
        %                               ... 'thet' = theta_total
        %                               ... 'thet*k' = theta_total * k
        % CDX2                          ... second CDX structure for robust RMSE calculation
    tic;
    cdx_demo = iterative_fitting(cdx_demo, discount_curves_demo, datenum('12/05/2005'), datenum('12/05/2005'), [1 0 0 1], [1 0 0], 2^18, ...
                                 20, 121, 0, ones(10, 1), 'mu*L', 'thet*k', cdx_demo);
    toc;    
    % REMARK: Use this call to optimize parameters for BOTH dates:
%     cdx_demo = iterative_fitting(cdx_demo, discount_curves_demo, datenum('12/05/2005'), datenum('01/05/2006'), [1 0 0 1], [1 0 0], 2^18, ...
%                                  2, 121, 0, ones(10, 1), 'mu*L', 'thet*k', cdx_demo);
    
    
    disp('* Determine new RMSE pricing error.*')
    [trash, cdx_demo] = wrapper_tranche_mispricing(get_x0(cdx_demo, [datenum('12/05/2005')]), cdx_demo, discount_curves_demo, datenum('12/05/2005'), ...
                                           datenum('12/05/2005'), 2^18, [1 0 0 1], [1 0 0], 121, 0, 2, 'mu*L', 'thet*k');
    rmse_total_new = cdx_demo.rmse(1);    
    disp(cdx_demo);
    disp(['After one iteration RMSE_total was reduced from ' num2str(rmse_total_orig) ' to ' num2str(rmse_total_new)])     
    h = msgbox({'Demonstrated parameter optimization.' '' 'Press OK to continue.'}, 'Step 1', 'none');
    uiwait(h);
end


disp('********************************************')
disp('* Demonstrate loading of fitted parameters *')
disp('********************************************')
disp(' ')

cdx_demo = load_fitted_model(cdx_demo, 'demo');
disp(cdx_demo)

h = msgbox({'Loaded fitted parameters from file ''fitted/demo.mat''.' '' 'Press OK to continue.'}, 'Step 2', 'none');
uiwait(h);


disp('*******************************************')
disp('* Demonstrate saving of fitted parameters *')
disp('*******************************************')
disp(' ')

save_fitted_model(cdx_demo, 'demo', datenum('12/05/2005'), datenum('12/06/2005'))

h = msgbox({'Save model parameters to file ''fitted/demo.mat''.' '' 'Press OK to continue.'}, 'Step 3', 'none');
uiwait(h);


