function [cds_new, rmse] = fit_model_cds5(cds, discounts_IMM, CDX, start_date_num, end_date_num, maturities)
% --------------------------------------------------------------------------------------------------
% Fit the factor loading (of the default intensity on common factor) and idiosyncratic intensity
% dynamics to the time-series of CDS prices.
% --------------------------------------------------------------------------------------------------
% cds               ... credit default swap structure (see 'all_steps_in_a_row.m')
% discounts_IMM     ... structure with discount curves (quarterly horizons)
% CDX               ... credit index structure which this CDX is part of (because need y0 & dynamics of common factor, liquidity premium)
% start_date_num    ... optional datenum of start date, default: use all dates
% end_date_num      ... optional datenum of end date, default: use all dates
% maturities        ... which maturities to include in calculation of RMSE, default: all
% --------------------------------------------------------------------------------------------------
% sample call: fit_model_cds5(CDX_NA_IG2.portfolio(1), discounts_cds, CDX_NA_IG2, datenum('03/17/2006'), datenum('03/31/2006'))
% --------------------------------------------------------------------------------------------------

% Set parameters for optimization
if (nargin <= 3)
    start_date_num = cds.dates{1}(1);
    end_date_num = cds.dates{1}(end);
end
if (nargin <= 5)
    maturities = ones(length(cds.dates), 1);
end

% Don't fit if no update desired
if (sum(maturities) == 0)
    cds_new = cds;
    return;
end

% If model is fit to only one maturity of CDS prices, then RMSE is zero => no optimization necessary
if ((maturities(1)==1) & (sum(maturities)==1))
    % Only update initial default intensity and model-implied price
    rmse = 0;
    cds_new = update_model_x0_v2(cds, discounts_IMM, CDX, start_date_num, end_date_num, 1);
    cds_new = update_model_price_cds(cds_new, discounts_IMM, maturities, start_date_num, end_date_num, CDX.liq_prem_cds);
    return;
end

% Determine date-range used
start_pos = find(cds.dates{1} >= start_date_num, 1, 'first');
end_pos = find(cds.dates{1} <= end_date_num, 1, 'last');
used_range = start_pos:end_pos;
used_dates = cds.dates{1}(used_range);
num_dates = length(used_dates);

% Use (possibly vectorized) interval search for value of ai
lower = repmat(1e-5, num_dates, 1);
upper = repmat(10, num_dates, 1);
[lower_mispricing, cds_new] = wrapper_CDS_mispricing2(lower, cds, CDX, discounts_IMM, start_date_num, end_date_num, maturities);
if (length(cds.ai) > 1) & (length(used_dates) > 1)
    lower_mispricing = cds_new.rmse;
end
[upper_mispricing, cds_new] = wrapper_CDS_mispricing2(upper, cds, CDX, discounts_IMM, start_date_num, end_date_num, maturities);
if (length(cds.ai) > 1) & (length(used_dates) > 1)
    upper_mispricing = cds_new.rmse;
end

% If no data in relevant date range, do nothing
if (lower_mispricing==0)
    cds_new = cds;
    rmse = 0;
    return;
end
    
% grid = (0.1:0.01:1)';
% values = zeros(length(num_dates), length(grid));
% for i=1:length(grid)
%     [total_mispricing, cds_new] = wrapper_CDS_mispricing2(repmat(grid(i), length(num_dates), 1), cds, CDX, discounts_IMM, start_date_num, end_date_num, maturities);
%     [trash, value_pos, cds_pos] = intersect_sorted(used_dates, cds.dates{1});
%     values(value_pos,i) = cds_new.rmse(cds_pos);
% end
% plot(grid, values);

[total_mispricing, cds_new] = wrapper_CDS_mispricing2(repmat(1e-3, length(num_dates), 1), cds, CDX, discounts_IMM, start_date_num, end_date_num, maturities);

precision = 1e-5;
while (max(upper - lower) > precision)
    % Compute derivative of mispricing error in weighted middle of search interval
    middle = (lower + upper)/2;
    [middle_mispricing, cds_middle] = wrapper_CDS_mispricing2(middle - precision/2, cds, CDX, discounts_IMM, start_date_num, end_date_num, maturities);
    [middle_mispricing2, cds_middle2] = wrapper_CDS_mispricing2(middle + precision/2, cds, CDX, discounts_IMM, start_date_num, end_date_num, maturities);
    
    if (length(lower) > 1)
        middle_mispricing = cds_middle.rmse(used_range);
        middle_mispricing2 = cds_middle2.rmse(used_range);
    end
    derivative = (middle_mispricing2 - middle_mispricing)/precision;
    %disp(['ai: ' num2str(middle)]);
    
    % Narrow down search interval
    positive = (derivative > 0);
    upper(positive) = middle(positive);
    upper_mispricing(positive) = middle_mispricing(positive) + middle_mispricing2(positive);
    lower(~positive) = middle(~positive);
    lower_mispricing(~positive) = middle_mispricing(~positive) + middle_mispricing2(~positive);        
end
ai = (lower+upper)/2;
%disp(['ai_new: ' num2str(ai)]);
            
% Update CDS with new parameters
[rmse, cds_new] = wrapper_CDS_mispricing2(ai, cds, CDX, discounts_IMM, start_date_num, end_date_num, maturities);
