function [cdx_new, rmse] = fit_model_cdx_xi4(i, CDX, discounts_IMM, start_date_num, end_date_num, N, maturities_cds, maturities_tranches, ...
                                             RMSE_method, LGD_method, how_port_loss, algo, method5, method2, x0_diff, CDX2)
% --------------------------------------------------------------------------------------------------
% Fit the whole model as good a possible (for the time series of cds and tranche prices) by just
% varying the i-th parameter.
% --------------------------------------------------------------------------------------------------
% i                             ... which parameter to vary (of time series of {k, theta, sigma, L_total, mu, omega1, omega2})
% CDX                           ... credit index structure (see 'all_steps_in_a_row.m')
% discounts_IMM                 ... structure with discount curves matching IMM dates
% start_date_num                ... datenum of start date
% end_date_num                  ... datenum of end date
% N                             ... number of points for numerical integration of Fourier transform
% maturities_cds                ... which maturities to fit for CDS prices, default: all
% maturities_tranches           ... which maturities to fit for tranche prices, default: all
% RMSE_method                   ... which definition of pricing error to use
% LGD_method                    ... method for (joint) distribution of LGDs
% how_port_loss                 ... how to calculate the portfolio loss distribution
%                                   1 ... load from external file
%                                   2 ... calculate
%                                   3 ... calculate and save in external file
% algo                          ... search algorithm:
%                                   1 ... interval search based on derivatives
%                                   2 ... classical interval search with shrinkage factor of 0.8
% method5                       ... which definition to use for the 5th parameters
%                                   1 = expected jump size \mu (default)
%                                   2 = jump intensity x jump size
% method2                       ... which definition to use for the 2nd parameter
%                               ... 'thet' = theta_total
%                               ... 'thet*k' = theta_total * k
% x0_diff                       ... change in i-th parameter of x0, last time this function was called
% CDX2                          ... second CDX structure for robust RMSE calculation (optional)
% --------------------------------------------------------------------------------------------------
% sample call: fit_model_cdx_xi4(i, CDX, discounts_IMM, datenum('02/01/2006'), datenum('02/01/2006'), 1000, [1 0 0 0], [1 0 0], 1, 0, 2, 2)
% --------------------------------------------------------------------------------------------------

if (nargin <= 15)
    CDX2 = [];
end

% Set speed of search interval shrinkage
alpha = 0.8;

% Get current parameters
start_pos = find(CDX.dates{1} >= start_date_num, 1, 'first');
end_pos = find(CDX.dates{1} <= end_date_num, 1, 'last');
used_dates = CDX.dates{1}(start_pos:end_pos);
x0 = get_x0(CDX, used_dates, method5, method2);
k = x0{1};
theta_total = x0{2};
sigma = x0{3};
if (nargin <= 14)
    x0_diff = ones(size(x0));
end

% Determine constraints: kappa*theta_Y > sigma^2/2 (Feller condition)
if (i==1)
    lower = repmat(-0.3, length(k), 1); %max(sigma.^2/2 ./ theta_total, 0.01);
    upper = repmat(0.1, length(lower), 1);
    lower = min(lower, upper);
elseif (i==2)
    if strcmp(method2, 'thet')
        lower = repmat(5e-4, length(theta_total), 1); %max(sigma.^2/2 ./ k, 0.01);
        upper = repmat(0.1, length(lower), 1);
    elseif strcmp(method2, 'thet*k')
        lower = 0.0005 .* abs(k);
        upper = 0.1 .* abs(k);
    end
    lower = min(lower, upper);
elseif (i==3)
    upper = repmat(0.15, length(sigma), 1); %min(sqrt(2*k .* theta_total), 0.1);
    lower = repmat(0.01, length(upper), 1);
    upper = max(upper, lower);
elseif (i==4)
    if strcmp(method5, 'mu')
        lower = 0.006;
        upper = 0.03;
    elseif strcmp(method5,'mu*L')
        lower = max(0.006, x0{5} / 0.3);
        upper = min(0.03, x0{5} / 0.01);        
    end
elseif (i==5)
    if strcmp(method5, 'mu')
        lower = 0.01;
        upper = 0.3;
    elseif strcmp(method5,'mu*L')
        lower = 0.01.*x0{4};
        upper = 0.3.*x0{4};
    end
else
    b = [0.05  0.05;
         1       1];
    lower = repmat(b(1, i-5), length(x0{i}), 1);
    upper = repmat(b(2, i-5), length(x0{i}), 1);
end
precision = [1e-4 1e-4 5e-5 2e-6 1e-3 1e-3 5e-3];
if strcmp(method5, 'mu*L')
    precision(5) = 1e-6;
end
if strcmp(method2, 'thet*k')
    precision(2) = 5e-7;
end

% Shrink search interval based on how much i-th parameter changed in last optimization
if (abs(x0_diff) < 1)
    % If close to convergence, use asymmetric search interval
    if (x0_diff > 0)
        range_lower = (-1) * max(precision(i)*2, abs(x0_diff)/2);
        range_upper = max(precision(i)*5, abs(x0_diff) * 2);
    else
        range_lower = (-1) * max(precision(i)*5, abs(x0_diff) * 2);
        range_upper = max(precision(i)*2, abs(x0_diff)/2);
    end
else
    range_lower = (-1) * max(precision(i)*5, abs(x0_diff) * 2);
    range_upper = max(precision(i)*5, abs(x0_diff) * 2);
end
lower = max(lower, x0{i} + range_lower);
upper = min(upper, x0{i} + range_upper);
upper = max(upper, lower);

% Define starting values for (vectorized) interval search
xi_old = x0{i};
is_ts_params = (length(xi_old) > 1);    % indicator whether xi is time-varying

% grid = (0.1:0.1:1)*upper;
% values = [];
% for j=1:length(grid)
%     [values_tmp, tmp] = wrapper_tranche_mispricing_xi(grid(j), i, CDX, discounts_IMM, start_date_num, end_date_num, ...
%                                                      3999, maturities_cds, maturities_tranches, RMSE_method, LGD_method, how_port_loss);
%     values(j) = sqrt(sum(values_tmp.^2) / length(values_tmp));                                     
%     if (j==1)
%         cdx_series = tmp;
%     else
%         cdx_series(j) = tmp;
%     end
% end
% plot(grid, values);

% Calculate RMSE at boundary of search interval
lower_mispricing = wrapper_tranche_mispricing_xi(lower, i, CDX, discounts_IMM, start_date_num, end_date_num, ...
                                                 N, maturities_cds, maturities_tranches, RMSE_method, LGD_method, how_port_loss, method5, method2, CDX2);
upper_mispricing = wrapper_tranche_mispricing_xi(upper, i, CDX, discounts_IMM, start_date_num, end_date_num, ...
                                                 N, maturities_cds, maturities_tranches, RMSE_method, LGD_method, how_port_loss, method5, method2, CDX2);
if (~is_ts_params)
    lower_mispricing = sqrt(sum(lower_mispricing.^2) / length(lower_mispricing));
    upper_mispricing = sqrt(sum(upper_mispricing.^2) / length(upper_mispricing));
end
 
% Algorithm I: use (vectorized) interval search for value of xi based on derivative
if (algo == 1)
    while (max(upper - lower) > precision(i))
        % Compute derivative of mispricing error in middle of search interval
        middle = (lower+upper)/2;
        [rms_tranches, trash, rms_5yr_cds] = wrapper_tranche_mispricing_xi(middle-precision(i)/2, i, CDX, discounts_IMM, ...
                                                  start_date_num, end_date_num, N, maturities_cds, maturities_tranches, RMSE_method, ...
                                                  LGD_method, how_port_loss, method5, method2, CDX2);
        [rms_tranches2, trash, rms_5yr_cds2] = wrapper_tranche_mispricing_xi(middle+precision(i)/2, i, CDX, discounts_IMM, ...
                                                    start_date_num, end_date_num, N, maturities_cds, maturities_tranches, RMSE_method, ...
                                                    LGD_method, how_port_loss, method5, method2, CDX2);
        if (~is_ts_params)
            middle_mispricing = sqrt(sum(rms_tranches.^2) / length(rms_tranches)) + 5*sqrt(sum(rms_5yr_cds.^2) / length(rms_5yr_cds));
            middle_mispricing2 = sqrt(sum(rms_tranches2.^2) / length(rms_tranches2)) + 5*sqrt(sum(rms_5yr_cds2.^2) / length(rms_5yr_cds2));
        else
            middle_mispricing = rms_tranches + 5*rms_5yr_cds;
            middle_mispricing2 = rms_tranches2 + 5*rms_5yr_cds2;
        end
        derivative = (middle_mispricing2-middle_mispricing) / precision(i);                                               
        disp(['derivative: ' num2str(derivative')]);

        % Check if algorithm isn't doing something stupid
        if (sum((min(middle_mispricing,middle_mispricing2)./lower_mispricing > 1 + 1e-4) & ...
            (min(middle_mispricing,middle_mispricing2)./upper_mispricing > 1 + 1e-4)) > 0)
            %error('Algorithm is doing something stupid!');
            disp('Algorithm is doing something stupid! Increasing number of grid points by 2000.');
            N = N + 2000;
            lower_mispricing = wrapper_tranche_mispricing_xi(lower, i, CDX, discounts_IMM, start_date_num, end_date_num, N, maturities_cds, ...
                                                             maturities_tranches, RMSE_method, LGD_method, how_port_loss, method5, method2, CDX2);
            upper_mispricing = wrapper_tranche_mispricing_xi(upper, i, CDX, discounts_IMM, start_date_num, end_date_num, N, maturities_cds, ...
                                                             maturities_tranches, RMSE_method, LGD_method, how_port_loss, method5, method2, CDX2);
            continue;
        end

        % Narrow down search interval
        pos_derivative = (derivative > 0);
        upper(pos_derivative) = middle(pos_derivative);
        upper_mispricing(pos_derivative) = (middle_mispricing(pos_derivative) + middle_mispricing2(pos_derivative))/2;
        lower(~pos_derivative) = middle(~pos_derivative);
        lower_mispricing(~pos_derivative) = (middle_mispricing(~pos_derivative) + middle_mispricing2(~pos_derivative))/2;
    end
end

% Algorithm II: use classical (vectorized) interval search (more stable)
if (algo == 2)
    while (max(upper - lower) > precision(i))
        % Compute mispricing error in weighted middle of search interval
        middle = zeros(length(lower), 1);
        lower_better = lower_mispricing <= upper_mispricing;
        middle(lower_better) = (1-alpha)*lower(lower_better) + alpha*upper(lower_better);
        middle(~lower_better) = alpha*lower(~lower_better) + (1-alpha)*upper(~lower_better);
        middle_mispricing = wrapper_tranche_mispricing_xi(middle, i, CDX, discounts_IMM, start_date_num, end_date_num, N, maturities_cds, ...
                                                          maturities_tranches, RMSE_method, LGD_method, how_port_loss, method5, method2, CDX2);

        % Narrow down search interval
        upper(lower_better) = middle(lower_better);
        upper_mispricing(lower_better) = middle_mispricing(lower_better);
        lower(~lower_better) = middle(~lower_better);
        lower_mispricing(~lower_better) = middle_mispricing(~lower_better);     
    end
end

% Update CDX with new parameters
xi_new = (lower+upper)/2;
x0{i} = xi_new;
disp(['x0: ' num2str(x0{i}')]);
[rmse, cdx_new] = wrapper_tranche_mispricing(x0, CDX, discounts_IMM, start_date_num, end_date_num, N, maturities_cds, ...
                                             maturities_tranches, RMSE_method, LGD_method, how_port_loss, method5, method2, CDX2);
