function [table, cds_names] = tranche_deltas_idio(CDX, date, discounts_IMM, N)
% --------------------------------------------------------------------------------------------------
% Calculate tranche deltas for the 125 possible changes of idiosyncratic default intensities.
% --------------------------------------------------------------------------------------------------
% CDX               ... credit index structure
% date              ... date for which to calculate tranche deltas
% discounts_IMM     ... structure with discount curves matching IMM dates
% N                 ... number of points for numerical integration of Fourier transform
% --------------------------------------------------------------------------------------------------
% sample call: tranche_deltas_idio(cdx_mor, '12/05/2005', discounts_IMM, 2^7)
% --------------------------------------------------------------------------------------------------

% Determine used CDS
date = datenum(date);
date_pos = find(CDX.dates{1} == date);
y0 = CDX.y0(date_pos);
used_cds = find(logical(CDX.index_members(date_pos,:)));
cds_names = {CDX.portfolio.company};
num_cds = length(used_cds);
delta_matrix = zeros(num_cds, length(CDX.cut_offs));

% Calculate tranche and index prices with orignal parameters
cdx_tmp = update_model_price_tranches(CDX, discounts_IMM, N, date, date, [1 0 0], 0, 2);
tranche_prices_orig = [cdx_tmp.tranche_model_upfront{1}(date_pos,1) cdx_tmp.tranche_model_price{1}(date_pos,2:end)]; 
cdx_tmp = update_model_price_index(CDX, discounts_IMM, [1 1 1], date, date);

bump_size = 1e-1;
for i=1:6 %num_cds
    % Bump initial idiosyncratic intensity of i-th CDS by one basis point
    disp(i);
    j = used_cds(i);
    cds = CDX.portfolio(j);
    date_pos_j = find(cds.dates{1} == date);
    if isempty(date_pos_j)
        continue;
    end
    cds_old_spread = cds.model_price{1}(date_pos_j);
    
    % Bump idiosyncratic intensity of i-th CDS
    cdx_tmp = bump_x0_i(CDX, date, bump_size, j);
    
    % Calculate new model-implied spread for CDS
    cds = cdx_tmp.portfolio(j);
    cds = update_model_price_cds(cds, discounts_IMM, [1 0 0], date, date, cdx_tmp.liq_prem_cds, cdx_tmp.AJD_common_factor, cdx_tmp.y0);
    cds_new_spread = cds.model_price{1}(date_pos_j);
    
    % Calculate tranche and index prices for bumped CDX
    cdx_tmp = update_model_price_tranches(cdx_tmp, discounts_IMM, N, date, date, [1 0 0], 0, 2);
    tranche_prices_new = [cdx_tmp.tranche_model_upfront{1}(date_pos,1) cdx_tmp.tranche_model_price{1}(date_pos,2:end)];                          
    cdx_tmp = update_model_price_index(cdx_tmp, discounts_IMM, [1 1 1], date, date);
    
    % Calculate delta for tranches (price change of tranches / price change of index)
    tranche_delta = (tranche_prices_new - tranche_prices_orig) / bump_size;
    tranche_delta(2:end) = tranche_delta(2:end) / 1e4;                      % delta in basis points running spread
    tranche_delta(1) = tranche_delta(1) / 100;                              % delta in percent up-front payment
    CDS_delta = (cds_new_spread - cds_old_spread) / bump_size / 1e4;
    delta_matrix(i,:) = tranche_delta / CDS_delta;
    delta_matrix(i,1) = delta_matrix(i,1) * 100 / cdx_tmp.PV01{1}(date_pos,1);                      
end
%disp(delta_matrix);

% Make nice table
cds_prices = zeros(num_cds, 1);
x0 = zeros(num_cds, 1);
y_syst = zeros(num_cds, 1);
for i=1:length(CDX.portfolio)   % Extract CDS prices & x_it at time 0
    cds = CDX.portfolio(i);
    if (~isempty(cds.market_price{1}))
        cds_prices(i) = cds.market_price{1}(date_pos);
        x0(i) = (cds.x0(date_pos) - y0 * cds.ai) * 1e4;
        y_syst(i) = y0 * cds.ai * 1e4;
    end
end
table = [cds_prices x0 y_syst delta_matrix];


