function [cdx_new, rmse_progress, x0_diff] = iterative_fitting(CDX, discounts_IMM, start_date_num, end_date_num, cds_maturities, CDX_maturities, ...
                                                               N, num_iter, RMSE_method, LGD_method, used_params, method5, method2, CDX2)
% --------------------------------------------------------------------------------------------------
% Iteratively fit ai (for CDS), y0, k, theta_total, sigma, L_total, mu, omega1, omega2.
% --------------------------------------------------------------------------------------------------
% RMSE_method                   ... which definition of pricing error to use
% LGD_method                    ... method for (joint) distribution of LGDs
% used_params                   ... indicator vector of length 10, which parameters to update
%                                   [k,theta,sigma,l,mu,omega1,omega2,y0,liq_prem_cds,liq_prem_tranches]
% method5                       ... which definition to use for the 5th parameter
%                                   'mu' = expected jump size \mu (default)
%                                   'mu*L' = jump intensity x jump size
% method2                       ... which definition to use for the 2nd parameter
%                               ... 'thet' = theta_total
%                               ... 'thet*k' = theta_total * k
% CDX2                          ... second CDX structure for robust RMSE calculation (optional)
% --------------------------------------------------------------------------------------------------
% sample call: iterative_fitting(CDX, discounts_IMM, ...)
% --------------------------------------------------------------------------------------------------

% Define some general fitting parameters
weight_cds_5yr = 5;     % weight of RMSE_CDS_5yr

% Determine date-range used
start_pos = find(CDX.dates{1} >= start_date_num, 1, 'first');
end_pos = find(CDX.dates{1} <= end_date_num, 1, 'last');
used_range = start_pos:end_pos;
used_dates = CDX.dates{1}(used_range);
x0_old = get_x0(CDX, used_dates, method5, method2);

% Initialize parameter changes from last iteration at large value
x0_diff = x0_old;
for i=1:length(x0_diff)
    x0_diff{i} = ones(size(x0_diff{i}));
end

% Iteratively optimize parameters
warning off all
variable_names = {'k' 'theta_total' 'sigma' 'L_total' 'mu' 'omega1' 'omega2'};
rmse_progress = [];
cdx_new = CDX;
for i=1:num_iter
    disp(' ');
    disp(['ITERATION ' num2str(i)])
    disp(' ');
    
    % Fit each parameter separately
    for j=1:length(variable_names)
         if (used_params(j) == 0)
            continue;
        end
        disp(' ');
        if (j == 2) & (strcmp(method2, 'thet*k'))
            text = 'theta_total*k';
        elseif (j == 5) & (strcmp(method5, 'mu*L'))
            text = 'mu*L';
        else
            text = variable_names{j};
        end
        disp(['Optimizing "' text '":']);
        disp(' ');
        cdx_old = cdx_new;               
        
        % Optimize j-th parameter
        cdx_new = fit_model_cdx_xi4(j, cdx_new, discounts_IMM, start_date_num, end_date_num, N, cds_maturities, ...
                                    CDX_maturities, RMSE_method, LGD_method, 2, 2, method5, method2, x0_diff{j}, CDX2);
        
        % Calculate pricing error for new parameters
        [trash, cdx_new] = wrapper_tranche_mispricing(get_x0(cdx_new, used_dates, method5, method2), cdx_new, discounts_IMM, start_date_num, end_date_num, N, ...
                                                      cds_maturities, CDX_maturities, RMSE_method, LGD_method, 2, method5, method2, CDX2);
        
        % For points in time where new parameters did not give an improvement, stick with old parameters
        no_improv = (cdx_old.rmse + weight_cds_5yr * cdx_old.rmse_5yr_cds) < (cdx_new.rmse + weight_cds_5yr * cdx_new.rmse_5yr_cds - 1e-4);
        if (sum(no_improv) > 0)         
            % For these dates, use old parameters, but only if old MU is in feasible range
%             if (j == 5)
%                 feasible = (cdx_old.AJD_common_factor.mu <= 0.25);
%                 no_improv = no_improv & feasible;
%             end
            no_improv_dates = cdx_new.dates{1}(no_improv);
            if (length(no_improv_dates) > 0)
                cdx_new = CDX_copy_parameters(cdx_new, cdx_old, no_improv_dates);
            end
        end
                                                                     
        % Track fitting progress
        rmse_progress = [rmse_progress cdx_new.rmse];
    end
    
    % Optimize y0
    if (used_params(8) == 1)
        disp(' ');
        disp('Optimizing y0:');
        disp(' ');
        cdx_old = cdx_new;
        cdx_new = update_model_y0_v2(cdx_new, discounts_IMM, N, start_date_num, end_date_num, cds_maturities, CDX_maturities, RMSE_method, ...
                                     LGD_method, 2, method5, method2, CDX2);
        % For points in time where new parameters did not give an improvement, stick with old parameters
        no_improv = (cdx_old.rmse + weight_cds_5yr * cdx_old.rmse_5yr_cds) < (cdx_new.rmse + weight_cds_5yr * cdx_new.rmse_5yr_cds - 1e-4);
        if (sum(no_improv) > 0)
            % Determine dates without improvement
            start_pos = find(CDX.dates{1} >= start_date_num, 1, 'first');
            end_pos = find(CDX.dates{1} <= end_date_num, 1, 'last');
            no_improv = no_improv(start_pos:end_pos);
            no_improv_dates = cdx_new.dates{1}(start_pos:end_pos);
            no_improv_dates = no_improv_dates(no_improv);
            cdx_new = CDX_copy_parameters(cdx_new, cdx_old, no_improv_dates);
        end
        rmse_progress = [rmse_progress cdx_new.rmse];
    end
    
    % Update liquidity premia estimate
    if (used_params(9) == 1)
        disp(' ');
        disp('Optimizing liq_prem_cds');
        disp(' ');
        cdx_new = fit_liq_prem_cds3(cdx_new, discounts_IMM, start_date_num, end_date_num, N, cds_maturities, CDX_maturities, RMSE_method, LGD_method, method5, method2, CDX2);
    end
    if (used_params(10) == 1)
        disp(' ');
        disp('Optimizing liq_prem_tranches');
        disp(' ');
        cdx_new = fit_liq_prem_tranches2(cdx_new, discounts_IMM, start_date_num, end_date_num, N, cds_maturities, CDX_maturities, RMSE_method, LGD_method, method5, method2, CDX2);
    end
    
    % Calculate change in parameters
    x0_new = get_x0(cdx_new, used_dates, method5, method2);
    x0_diff = {};
    for k=1:length(x0_new)
        x0_diff{k} = x0_new{k} - x0_old{k};
    end
    x0_old = x0_new;
end
if (~isempty(rmse_progress))
    plot(rmse_progress');
end

