function cds_fractions = cds_spread_decomposition(i, CDX, discounts_IMM, date, maturities_cds, LGD_method, theta_P, param_type)
% --------------------------------------------------------------------------------------------------
% Decompose 5-year CDS spread (of i-th issuer in CDX portfolio) into different components like:
% compensation for expected losses, JTD risk premium, correlation risk premium, MTM risk premium,...
% --------------------------------------------------------------------------------------------------
% i                             ... which CDS spread in portfolio to decompose
% CDX                           ... credit index structure (see 'all_steps_in_a_row.m')
% discounts_IMM                 ... structure with discount curves matching IMM dates
% date                          ... date for which to do decomposition
% maturities_cds                ... which maturities to fit for CDS prices, default: all
% LGD_method                    ... method for (joint) distribution of LGDs
% theta_P                       ... vector physical parameters
% param_type                    ... type of physical parameters
%                                   1 ... (k, L, mu)
%                                   2 ... (k, L, omega4)        where mu_i^P = omega4 * X_it
%                                   3 ... (k, omega5, omega4)   where mu_i^P = omega4 * X_it and l_i^P = omega5 * X_it
% --------------------------------------------------------------------------------------------------
% sample call: cds_spread_decomposition(1, cdx_tv_monthly10, discounts_IMM, datenum('11/01/2006'), [1 1 1 1], 0, theta_P, 1)
% --------------------------------------------------------------------------------------------------

% Extract CDS
cds = CDX.portfolio(i);

% Determine date position in CDX structure
business_cycle_length = 6;
date = datenum(date);
date_pos = find(cds.dates{1} == date);
date_pos_cdx = find(CDX.dates{1} == date);

% Update model-implied CDS spread under risk-neutral measure, and store for later on
cds = update_model_price_cds(cds, discounts_IMM, maturities_cds, date, date, CDX.liq_prem_cds(date_pos_cdx), CDX.AJD_common_factor, CDX.y0);
cds_spread = cds.model_price{1}(date_pos);

% Set all risk premia equal to zero => calculate expected losses
CDX_tmp = CDX;
% MTM risk premium for systematic factor
CDX = set_zero_RP_MPR_syst(CDX, business_cycle_length);
% MTM risk premium for idiosyncratic factors
CDX = set_zero_RP_MPR_idio(CDX, theta_P(1));
% Jump risk premium for idiosyncratic factors
CDX = set_zero_RP_J_idio(CDX, theta_P(2), theta_P(3), param_type);
% JTD risk premium & correlation risk premium
CDX = set_zero_RP_pJTD_and_COR(CDX);
% Update prices CDS price
cds = CDX.portfolio(i);
cds = update_model_price_cds(cds, discounts_IMM, maturities_cds, date, date, CDX.liq_prem_cds(date_pos_cdx), CDX.AJD_common_factor, CDX.y0);
cds_spread = [cds_spread; cds.model_price{1}(date_pos)];
CDX = CDX_tmp; 

% Set MTM risk premium for systematic factor equal to zero, i.e. k_Y_Q  <-  k_Y_P
CDX_tmp = CDX;
CDX = set_zero_RP_MPR_syst(CDX, business_cycle_length);
% Update prices CDS price
cds = CDX.portfolio(i);
cds = update_model_price_cds(cds, discounts_IMM, maturities_cds, date, date, CDX.liq_prem_cds(date_pos_cdx), CDX.AJD_common_factor, CDX.y0);
cds_spread = [cds_spread; cds.model_price{1}(date_pos)];
CDX = CDX_tmp; 

% Set MTM and Jump risk premium for idiosyncratic factors equal to zero
CDX_tmp = CDX;
CDX = set_zero_RP_MPR_idio(CDX, theta_P(1));
CDX = set_zero_RP_J_idio(CDX, theta_P(2), theta_P(3), param_type);
% Update prices CDS price
cds = CDX.portfolio(i);
cds = update_model_price_cds(cds, discounts_IMM, maturities_cds, date, date, CDX.liq_prem_cds(date_pos_cdx), CDX.AJD_common_factor, CDX.y0);
cds_spread = [cds_spread; cds.model_price{1}(date_pos)];
CDX = CDX_tmp;

% Set JTD risk premium equal to zero
CDX_tmp = CDX;
CDX = set_zero_RP_pJTD(CDX);
% Update prices CDS price
cds = CDX.portfolio(i);
cds = update_model_price_cds(cds, discounts_IMM, maturities_cds, date, date, CDX.liq_prem_cds(date_pos_cdx), CDX.AJD_common_factor, CDX.y0);
cds_spread = [cds_spread; cds.model_price{1}(date_pos)];
CDX = CDX_tmp;        
        
% Set correlation risk premium equal to zero, i.e. a_it_Q <- a_it_P * (b_Q / b_P)
CDX_tmp = CDX;
CDX = set_zero_RP_COR(CDX);
% Update prices CDS price
cds = CDX.portfolio(i);
cds = update_model_price_cds(cds, discounts_IMM, maturities_cds, date, date, CDX.liq_prem_cds(date_pos_cdx), CDX.AJD_common_factor, CDX.y0);
cds_spread = [cds_spread; cds.model_price{1}(date_pos)];
CDX = CDX_tmp;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Calculate contribution of individual risk premia %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Calculate CDS spread fractions
cds_fractions = cds_spread(2) / cds_spread(1);
remaining = 1 - cds_fractions(1);
tmp = (cds_spread(1)-cds_spread(3:end));
tmp = tmp/sum(tmp);
cds_fractions = [cds_fractions; tmp*remaining];

% Plot decomposition of index and tranche spreads (bar graphs)
data = [cds_fractions ones(5, 1)/5];
bar1 = bar(data', 'stack', 'BarWidth', 0.6);
set(gca,['x','ticklabel'], {CDX.portfolio(i).ticker});
h = findobj(gca,'Type','patch');
%applyhatch(gcf, '\|+./')
ylabel('fraction');
%colormap gray
colormap pink;
set(bar1(1), 'FaceColor', [0.33 0.33 0.33]);
legend({'Expected Loss' 'MPR Sytematic' 'MPR Idiosyncratic' 'Pure JTD RP' 'Correlation RP'}, 'Location', 'NorthEast');
axis([0.5 1.5 0 1.00000001])


