function [tranche_fractions, index_fractions, frac_average, frac_min, frac_max, frac_std, idx_frac_average, idx_frac_std] = ...
        table_tranche_spread_decomp(CDX, discounts_IMM, N, maturities_cds, maturities_tranches, LGD_method, theta_P, param_type, attribution_method)
% --------------------------------------------------------------------------------------------------
% Determine tranche spread decomposition at each point in time, and creat summary table
% --------------------------------------------------------------------------------------------------
% CDX                           ... credit index structure (see 'all_steps_in_a_row.m')
% discounts_IMM                 ... structure with discount curves matching IMM dates
% N                             ... number of points for numerical integration of Fourier transform
% maturities_cds                ... which maturities to fit for CDS prices, default: all
% maturities_tranches           ... which maturities to fit for tranche prices, default: all
% LGD_method                    ... method for (joint) distribution of LGDs
% theta_P                       ... vector physical parameters
% param_type                    ... type of physical parameters
%                                   1 ... (k, L, mu)
%                                   2 ... (k, L, omega4)        where mu_i^P = omega4 * X_it
%                                   3 ... (k, omega5, omega4)   where mu_i^P = omega4 * X_it and l_i^P = omega5 * X_it
% attribution_method            ... what decomposition to use
%                                   1 ... tranche spreads
%                                   2 ... losses
% --------------------------------------------------------------------------------------------------
% sample call: table_tranche_spread_decomp(cdx_tv_monthly10, discounts_IMM, 2^18, [1 1 1 0], [1 1 1], 0, theta_P, 1)
% --------------------------------------------------------------------------------------------------

% Allocate memory
num_dates = length(CDX.dates{1});
num_tranches = length(CDX.cut_offs);
tranche_fractions = zeros(num_dates, 5, num_tranches);
index_fractions = zeros(num_dates, 5);

% Determine tranche spread decomposition at each point in time
for i=1:num_dates
    date = CDX.dates{1}(i);
    [index_frac, tranche_frac, idx_loss, tranche_loos] = tranche_spread_decomposition(CDX, discounts_IMM, date, N, maturities_cds, maturities_tranches, ...
                                                         LGD_method, theta_P, param_type);
    if (attribution_method == 1)
        tranche_fractions(i,:,:) = tranche_frac;
        index_fractions(i,:) = index_frac;
    elseif (attribution_method == 2)
        tranche_fractions(i,:,:) = tranche_loos;
        index_fractions(i,:) = idx_loss;
    end
end

% Cap contribution at 0 and 1, and re-normalize
if (attribution_method == 1)
    tranche_fractions = min(1, max(0, tranche_fractions));
    index_fractions = min(1, max(0, index_fractions));
    for i=1:num_dates
        index_fractions(i,:) = index_fractions(i,:) / sum(index_fractions(i,:));
        for j=1:num_tranches
            tranche_fractions(i,:,j) = tranche_fractions(i,:,j) / sum(tranche_fractions(i,:,j));
        end
    end
end

% Create summary tables
frac_average = squeeze(mean(tranche_fractions, 1));
frac_min = squeeze(min(tranche_fractions, [], 1));
frac_max = squeeze(max(tranche_fractions, [], 1));
frac_std = squeeze(std(tranche_fractions, 1));
idx_frac_average = mean(index_fractions, 1);
idx_frac_std = std(index_fractions, 1);

% Compute row and column sum time series
tranche_total_frac = squeeze(sum(tranche_fractions, 2));
factor_total_frac = squeeze(sum(tranche_fractions, 3));
disp(['Factor total means: ' num2str(round(mean(factor_total_frac)*1000)/1000)])
disp(['Factor total standard deviations: ' num2str(round(std(factor_total_frac)*1000)/1000)])
disp(['Tranche total sums: ' num2str(round(mean(tranche_total_frac)*1000)/1000)])
disp(['Tranche total standard deviations: ' num2str(round(std(tranche_total_frac)*1000)/1000)])

% Total fraction time series
total_frac = sum(tranche_total_frac,2);
disp(['Total fraction mean: ' num2str(round(mean(total_frac)*1000)/1000)])
disp(['Total fraction standard deviations: ' num2str(round(std(total_frac)*1000)/1000)])


