function [index_fractions, tranche_fractions, mv_index_loss_fractions, mv_tranche_loss_fractions] = tranche_spread_decomposition(CDX, discounts_IMM, date, ...
         N, maturities_cds,  maturities_tranches, LGD_method, theta_P, param_type)
% --------------------------------------------------------------------------------------------------
% Decompose 5-year tranche spreads into different components like: compensation for expected losses,
% JTD risk premium, correlation risk premium, MTM risk premium, ...
% --------------------------------------------------------------------------------------------------
% CDX                           ... credit index structure (see 'all_steps_in_a_row.m')
% discounts_IMM                 ... structure with discount curves matching IMM dates
% date                          ... date for which to do decomposition
% N                             ... number of points for numerical integration of Fourier transform
% maturities_cds                ... which maturities to fit for CDS prices, default: all
% maturities_tranches           ... which maturities to fit for tranche prices, default: all
% LGD_method                    ... method for (joint) distribution of LGDs
% theta_P                       ... vector physical parameters
% param_type                    ... type of physical parameters
%                                   1 ... (k, L, mu)
%                                   2 ... (k, L, omega4)        where mu_i^P = omega4 * X_it
%                                   3 ... (k, omega5, omega4)   where mu_i^P = omega4 * X_it and l_i^P = omega5 * X_it
% --------------------------------------------------------------------------------------------------
% sample call: tranche_spread_decomposition(cdx_tv_monthly10, discounts_IMM, date, 2^18, [1 1 1 0], [1 1 1], 0, theta_P)
% --------------------------------------------------------------------------------------------------

business_cycle_length = 6;
method5 = 'mu*L';
method2 = 'thet*k';

% Update model-implied tranche spreads and index spread under risk-neutral measure, and store for
% later on
CDX.liq_prem_tranches = zeros(size(CDX.liq_prem_tranches));
date = datenum(date);
[trash, CDX] = wrapper_tranche_mispricing(get_x0(CDX, date, method5, method2), CDX, discounts_IMM, date, date, N, ...
                                          maturities_cds, maturities_tranches, 1, LGD_method, 2, method5, method2);
CDX = update_model_price_index(CDX, discounts_IMM, maturities_tranches, date, date);                                      
date_pos = find(CDX.dates{1} == date);
index_spreads = CDX.index_model_price{1}(date_pos);
tranche_spreads = [CDX.tranche_model_upfront{1}(date_pos,1) + 0.05 * CDX.PV01{1}(date_pos,1) ...
                   CDX.tranche_model_price{1}(date_pos,2:end)];
mv_loss = [tranche_spreads(1)  tranche_spreads(2:end) .* CDX.PV01{1}(date_pos,2:end) / 1e4];
mv_loss = mv_loss .* diff([0 CDX.cut_offs]);
mv_index_loss = CDX.index_model_price{1}(date_pos) * CDX.index_PV01{1}(date_pos) / 1e4;

% Set all risk premia equal to zero => calculate expected losses
CDX_tmp = CDX;
% MTM risk premium for systematic factor
CDX = set_zero_RP_MPR_syst(CDX, business_cycle_length);
% MTM risk premium for idiosyncratic factors
CDX = set_zero_RP_MPR_idio(CDX, theta_P(1));
% Jump risk premium for idiosyncratic factors
CDX = set_zero_RP_J_idio(CDX, theta_P(2), theta_P(3), param_type);
% JTD risk premium & correlation risk premium
CDX = set_zero_RP_pJTD_and_COR(CDX);
% Update prices
CDX = update_model_price_index(CDX, discounts_IMM, maturities_tranches, date, date);
CDX = update_model_price_tranches(CDX, discounts_IMM, N, date, date, maturities_tranches, LGD_method, 2);
index_spreads = [index_spreads; CDX.index_model_price{1}(date_pos)];
tranche_spreads = [tranche_spreads; [CDX.tranche_model_upfront{1}(date_pos,1) + 0.05 * CDX.PV01{1}(date_pos,1) ...
                   CDX.tranche_model_price{1}(date_pos,2:end)]];
tmp = [tranche_spreads(end,1)  tranche_spreads(end,2:end) .* CDX.PV01{1}(date_pos,2:end) / 1e4];
tmp = tmp .* diff([0 CDX.cut_offs]);  
mv_loss = [mv_loss; tmp];
mv_index_loss = [mv_index_loss; CDX.index_model_price{1}(date_pos) * CDX.index_PV01{1}(date_pos) / 1e4];
CDX = CDX_tmp; 

% Set MTM risk premium for systematic factor equal to zero, i.e. k_Y_Q  <-  k_Y_P
CDX_tmp = CDX;
CDX = set_zero_RP_MPR_syst(CDX, business_cycle_length);
CDX = update_model_price_index(CDX, discounts_IMM, maturities_tranches, date, date);
CDX = update_model_price_tranches(CDX, discounts_IMM, N, date, date, maturities_tranches, LGD_method, 2);
index_spreads = [index_spreads; CDX.index_model_price{1}(date_pos)];
tranche_spreads = [tranche_spreads; [CDX.tranche_model_upfront{1}(date_pos,1) + 0.05 * CDX.PV01{1}(date_pos,1) ...
                   CDX.tranche_model_price{1}(date_pos,2:end)]];
tmp = [tranche_spreads(end,1)  tranche_spreads(end,2:end) .* CDX.PV01{1}(date_pos,2:end) / 1e4];
tmp = tmp .* diff([0 CDX.cut_offs]);  
mv_loss = [mv_loss; tmp];               
mv_index_loss = [mv_index_loss; CDX.index_model_price{1}(date_pos) * CDX.index_PV01{1}(date_pos) / 1e4];
CDX = CDX_tmp; 

% Set MTM and Jump risk premium for idiosyncratic factors equal to zero
CDX_tmp = CDX;
CDX = set_zero_RP_MPR_idio(CDX, theta_P(1));
CDX = set_zero_RP_J_idio(CDX, theta_P(2), theta_P(3), param_type);
CDX = update_model_price_index(CDX, discounts_IMM, maturities_tranches, date, date);
CDX = update_model_price_tranches(CDX, discounts_IMM, N, date, date, maturities_tranches, LGD_method, 2);
index_spreads = [index_spreads; CDX.index_model_price{1}(date_pos)];
tranche_spreads = [tranche_spreads; [CDX.tranche_model_upfront{1}(date_pos,1) + 0.05 * CDX.PV01{1}(date_pos,1) ...
                   CDX.tranche_model_price{1}(date_pos,2:end)]];
tmp = [tranche_spreads(end,1)  tranche_spreads(end,2:end) .* CDX.PV01{1}(date_pos,2:end) / 1e4];
tmp = tmp .* diff([0 CDX.cut_offs]);  
mv_loss = [mv_loss; tmp];   
mv_index_loss = [mv_index_loss; CDX.index_model_price{1}(date_pos) * CDX.index_PV01{1}(date_pos) / 1e4];
CDX = CDX_tmp;

% Set JTD risk premium equal to zero
CDX_tmp = CDX;
CDX = set_zero_RP_pJTD(CDX);
% Update prices
CDX = update_model_price_index(CDX, discounts_IMM, maturities_tranches, date, date);
CDX = update_model_price_tranches(CDX, discounts_IMM, N, date, date, maturities_tranches, LGD_method, 2);
index_spreads = [index_spreads; CDX.index_model_price{1}(date_pos)];
tranche_spreads = [tranche_spreads; [CDX.tranche_model_upfront{1}(date_pos,1) + 0.05 * CDX.PV01{1}(date_pos,1) ...
                   CDX.tranche_model_price{1}(date_pos,2:end)]];
tmp = [tranche_spreads(end,1)  tranche_spreads(end,2:end) .* CDX.PV01{1}(date_pos,2:end) / 1e4];
tmp = tmp .* diff([0 CDX.cut_offs]);  
mv_loss = [mv_loss; tmp];    
mv_index_loss = [mv_index_loss; CDX.index_model_price{1}(date_pos) * CDX.index_PV01{1}(date_pos) / 1e4];
CDX = CDX_tmp;        
        
% Set correlation risk premium equal to zero, i.e. a_it_Q <- a_it_P * (b_Q / b_P)
CDX_tmp = CDX;
CDX = set_zero_RP_COR(CDX);
CDX = update_model_price_index(CDX, discounts_IMM, maturities_tranches, date, date);
CDX = update_model_price_tranches(CDX, discounts_IMM, N, date, date, maturities_tranches, LGD_method, 2);
index_spreads = [index_spreads; CDX.index_model_price{1}(date_pos)];
tranche_spreads = [tranche_spreads; [CDX.tranche_model_upfront{1}(date_pos,1) + 0.05 * CDX.PV01{1}(date_pos,1) ...
                   CDX.tranche_model_price{1}(date_pos,2:end)]];
tmp = [tranche_spreads(end,1)  tranche_spreads(end,2:end) .* CDX.PV01{1}(date_pos,2:end) / 1e4];
tmp = tmp .* diff([0 CDX.cut_offs]);  
mv_loss = [mv_loss; tmp];          
mv_index_loss = [mv_index_loss; CDX.index_model_price{1}(date_pos) * CDX.index_PV01{1}(date_pos) / 1e4];
CDX = CDX_tmp;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Calculate contribution of individual risk premia %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Calculate index fractions
index_fractions = index_spreads(2) / index_spreads(1);
remaining = 1 - index_fractions(1);
tmp = (index_spreads(1)-index_spreads(3:end));
tmp = tmp/sum(tmp);
index_fractions = [index_fractions; tmp*remaining];

% Calculate tranche fractions
tranche_fractions = zeros(size(tranche_spreads, 1)-1, size(tranche_spreads, 2));
tranche_fractions(1,:) = tranche_spreads(2,:) ./ tranche_spreads(1,:);
remaining = ones(1,size(tranche_spreads, 2)) -  tranche_fractions(1,:);
tmp =  repmat(tranche_spreads(1,:), size(tranche_spreads,1)-2, 1) - tranche_spreads(3:end,:);
tmp = tmp ./ repmat(sum(tmp, 1), size(tranche_spreads,1)-2, 1);
tranche_fractions(2:end,:) = tmp .* repmat(remaining, size(tranche_spreads,1)-2, 1);

% Calculate market_value of index loss fractions
total_loss = mv_index_loss(1);
mv_index_loss_fractions = mv_index_loss(2) / total_loss;
remaining = 1 - mv_index_loss_fractions;
index_losses = mv_index_loss(1) - mv_index_loss(3:end);
index_losses_frac = index_losses / sum(index_losses) * remaining;
mv_index_loss_fractions = [mv_index_loss_fractions; index_losses_frac];

% Calculate market_value of tranche loss fractions
mv_tranche_loss_fractions = mv_loss(2,:) / total_loss;
remaining = sum(mv_loss(1,:) - mv_loss(2,:)) / total_loss;
losses = (repmat(mv_loss(1,:), 4, 1) - mv_loss(3:end,:));
loss_frac = losses / sum(sum(losses)) * remaining;
mv_tranche_loss_fractions = [mv_tranche_loss_fractions; loss_frac];

% Plot decomposition of index and tranche spreads (bar graphs)
data = [index_fractions tranche_fractions];
bar1 = bar(data', 'stack', 'BarWidth',0.6);
set(gca,['x','ticklabel'], {'Index' 'Equity' 'JunMezz' 'Mezz' 'Senior' 'SupSenior'});
h = findobj(gca,'Type','patch');
ylabel('fraction');
%colormap gray
colormap pink;
set(bar1(1), 'FaceColor', [0.33 0.33 0.33]);
legend({'Expected Loss' 'MPR Sytematic' 'MPR Idiosyncratic' 'Pure JTD RP' 'Correlation RP'}, 'Location', 'NorthEast');
axis([0.5 6.5 0 1.00000001])


