function liq_error = wrapper_liq_cds_mispricing(liq_prem_cds, CDX, discounts_IMM, start_date_num, end_date_num, maturities_cds, maturities_tranches)
% --------------------------------------------------------------------------------------------------
% Wrapper function that calculates the pricing error of credit index vs. cds. The error is a sum of
% the errors for different maturities.
% --------------------------------------------------------------------------------------------------
% liq_prem_cds                  ... liquidity premium of CDS vs. index
% CDX                           ... credit index structure (see 'all_steps_in_a_row.m')
% discounts_IMM                 ... structure with discount curves matching IMM dates
% start_date_num                ... datenum of start date
% end_date_num                  ... datenum of end date
% N                             ... number of points for numerical integration of Fourier transform
% maturities_cds                ... which maturities to fit for CDS prices, default: all
% maturities_tranches           ... which maturities to take into account for calculating liq_error
% --------------------------------------------------------------------------------------------------
% sample call: wrapper_liq_cds_mispricing(cdx_mor, discounts_IMM, datenum('12/05/2005'), datenum('12/05/2005'), 10000, [1 0 0 1], [1 0 0], 1, 0)
% --------------------------------------------------------------------------------------------------

% Determine relevant date range - 5yr
start_pos = find(CDX.dates{1} >= start_date_num, 1, 'first');
end_pos = find(CDX.dates{1} <= end_date_num, 1, 'last');
used_range = start_pos:end_pos;
used_dates = CDX.dates{1}(used_range);

% Determine relevant date range - 7yr
available_5yr = (CDX.index_market_price{1}(used_range) > 0);
start_pos = find(CDX.dates{2} >= start_date_num, 1, 'first');
end_pos = find(CDX.dates{2} <= end_date_num, 1, 'last');
used_range2 = start_pos:end_pos;
used_dates2 = CDX.dates{2}(used_range2);
available_7yr = (CDX.index_market_price{2}(used_range2) > 0);

% Determine relevant date range - 10yr
start_pos = find(CDX.dates{3} >= start_date_num, 1, 'first');
end_pos = find(CDX.dates{3} <= end_date_num, 1, 'last');
used_range3 = start_pos:end_pos;
used_dates3 = CDX.dates{3}(used_range3);
available_10yr = (CDX.index_market_price{3}(used_range3) > 0);

% Calculate index and CDS spreads
CDX.liq_prem_cds(used_range, :) = liq_prem_cds;
CDX = fit_model_cds_portfolio(CDX, discounts_IMM, start_date_num, end_date_num, maturities_cds);
CDX = update_model_price_index(CDX, discounts_IMM, maturities_tranches, start_date_num, end_date_num);

if (maturities_tranches(1) == 1)
    % Calculate average CDS pricing bias for corresponding maturity
    avg_bias = plot_cds_bias(CDX, 1, 0, 0, used_dates);
    
    % Calculate liquidity error adjusted for CDS bias
    liq_error_5yr = zeros(length(used_range), 1);
    liq_error_5yr(available_5yr) = CDX.index_model_price{1}(used_range(available_5yr)) - CDX.index_market_price{1}(used_range(available_5yr));
    liq_error_5yr(available_5yr) = liq_error_5yr(available_5yr) -  avg_bias(available_5yr);  
end
if (maturities_tranches(2) == 1)
     % Calculate average CDS pricing bias for corresponding maturity
    avg_bias = plot_cds_bias(CDX, 2, 0, 0, used_dates2);
    
    % Calculate liquidity error adjusted for CDS bias
    % PV01)
    liq_error_7yr = zeros(length(used_range2), 1);
    liq_error_7yr(available_7yr) = CDX.index_model_price{2}(used_range2(available_7yr)) - CDX.index_market_price{2}(used_range2(available_7yr));
    liq_error_7yr(available_7yr) = liq_error_7yr(available_7yr) -  avg_bias(available_7yr);  
end
if (maturities_tranches(3) == 1)
    % Calculate average CDS pricing bias for corresponding maturity
    avg_bias = plot_cds_bias(CDX, 3, 0, 0, used_dates3);
    
    % Calculate liquidity error adjusted for CDS bias
    liq_error_10yr = zeros(length(used_range3), 1);
    liq_error_10yr(available_10yr) = CDX.index_model_price{3}(used_range3(available_10yr)) - CDX.index_market_price{3}(used_range3(available_10yr));
    liq_error_10yr(available_10yr) = liq_error_10yr(available_10yr) - avg_bias(available_10yr);  
end

% Sum up errors
liq_error = zeros(length(used_range), 1);
if (maturities_tranches(1) == 1)
    liq_error = liq_error + liq_error_5yr;
end
if (maturities_tranches(2) == 1)
    [trash, pos_5yr, pos_7yr] = intersect_sorted(used_dates, used_dates2);
    liq_error(pos_5yr) = liq_error(pos_5yr) + liq_error_7yr(pos_7yr);
end
if (maturities_tranches(3) == 1)
    [trash, pos_5yr, pos_10yr] = intersect_sorted(used_dates, used_dates3);
    liq_error(pos_5yr) = liq_error(pos_5yr) + liq_error_10yr(pos_10yr);
end

    
    
        
