function [rho_n, CDX] = optimize_rho_n(CDX, discounts_IMM, start_date_num, end_date_num, N, maturities_cds, maturities_tranches, ...
                                             RMSE_method, step_size, method5, method2, CDX2)
% --------------------------------------------------------------------------------------------------
% Optimize parameters rho_n that determine G_n in method 3 of LGD generation.
% --------------------------------------------------------------------------------------------------
% CDX                           ... credit index structure (see 'all_steps_in_a_row.m')
% discounts_IMM                 ... structure with discount curves matching IMM dates
% start_date_num                ... datenum of start date
% end_date_num                  ... datenum of end date
% N                             ... number of points for numerical integration of Fourier transform
% maturities_cds                ... which maturities to fit for CDS prices, default: all
% maturities_tranches           ... which maturities to fit for tranche prices, default: all
% RMSE_method                   ... which definition of pricing error to use
% step_size                     ... step size for search of optimal alphas
% method5                       ... which definition to use for the 5th parameters
%                                   'mu' = expected jump size \mu (default)
%                                   'mu*L' = jump intensity x jump size
% method2                       ... which definition to use for the 2nd parameter
%                               ... 'thet' = theta_total
%                               ... 'thet*k' = theta_total * k
% CDX2                          ... second CDX structure for robust RMSE calculation (optional)
% --------------------------------------------------------------------------------------------------
% sample call:
%{
optimize_rho_n(cdx_new1, discounts_IMM, datenum('02/01/2006'), datenum('02/01/2006'), 2000, [1 0 0 0], [1 0 0], 1)
%}
% --------------------------------------------------------------------------------------------------

if (nargin <= 11)
    CDX2 = [];
end

% Generate starting values
rho_n = CDX.LGD.rho_n;
old_rho_n = repmat(-1, size(rho_n));

% Calculate portfolio loss distribution up-front and let it be saved in an external file
[trash, CDX] = wrapper_tranche_mispricing(get_x0(CDX), CDX, discounts_IMM, start_date_num, end_date_num, N, ...
                                          maturities_cds, maturities_tranches, RMSE_method, 3, 3, method5, method2, CDX2);
rmse = CDX.rmse;                                  
             
% Optimize over rho_n   
counter = 1;
while (max(abs(rho_n - old_rho_n)) > 1e-3)
    disp(['Iteration: ' num2str(counter)]);
    disp(['Average rho_n: ' num2str(mean(rho_n))]);
    %disp(max(rho_n - old_rho_n));
    counter = counter + 1;
    old_rho_n = rho_n;
    
    % Increasing rho_n
    for j=1:length(rho_n)
        % Calculate RMSE for trial value of rho_j
        rho_try = rho_n;
        if (rho_try(j) == 1)
            continue;
        end
        rho_try(j) = min(rho_try(j) + step_size, 1);
        CDX.LGD.rho_n = rho_try;
        [rmse_new, CDX] = wrapper_tranche_mispricing(get_x0(CDX), CDX, discounts_IMM, start_date_num, end_date_num, N, ...
                                                     maturities_cds, maturities_tranches, RMSE_method, 3, 1, method5, method2, CDX2);
                                                 
        % If new RMSE is smaller than old one, use this value for rho_j
        if (rmse_new < rmse)
            rmse = rmse_new;
            rho_n = rho_try;
        end
    end
    
    % Decreasing rho_n
    for j=1:length(rho_n)
        % Calculate RMSE for trial value of rho_j
        rho_try = rho_n;
        if (rho_try(j) == 0)
            continue;
        end
        rho_try(j) = max(rho_try(j) - step_size, 0);
        CDX.LGD.rho_n = rho_try;
        [rmse_new, CDX] = wrapper_tranche_mispricing(get_x0(CDX), CDX, discounts_IMM, start_date_num, end_date_num, N, ...
                                                     maturities_cds, maturities_tranches, RMSE_method, 3, 1, method5, method2, CDX2);
                                                 
        % If new RMSE is smaller than old one, use this value for rho_j
        if (rmse_new < rmse)
            rmse = rmse_new;
            rho_n = rho_try;
        end
    end
end
CDX.LGD.rho_n = rho_n;
plot(rho_n)
%plot_rho_n(rho_n);
