function plot_fitted_intensities(CDX, ticker, maturity, ind_plots)
% --------------------------------------------------------------------------------------------------
% Plot fitted default intensities (under P and Q) for a company
% --------------------------------------------------------------------------------------------------
% CDS       ... credit index structure
% ticker    ... ticker symbol for ocmpany of interest
% maturity  ... which CDS maturity to plot
% ind_plots ... indicator of length five, which intensities to plot, default: [1 1 1 0 0]
% --------------------------------------------------------------------------------------------------
% sample call: plot_fitted_intensities(CDX_NA_IG2, 'AA', 5)
% --------------------------------------------------------------------------------------------------

if (nargin <= 3)
    ind_plots = [1 1 1 0 0];
end

% Extract relevant CDS
all_tickers = {CDX.portfolio.ticker};
pos = strcmp(all_tickers, ticker);
cds = CDX.portfolio(pos);

% Plot CDS spread
tmp = [5 7 10 1];
maturity_pos = find(tmp == maturity);
dates_scale = datenum(cds.dates{maturity_pos})-datenum(cds.dates{maturity_pos}(1));
if (ind_plots(1) == 1)
    semilogy(dates_scale, cds.market_price{maturity_pos}/cds.LGD, 'LineWidth', 1.2)
end

% Plot lambda_P
hold all
if (ind_plots(2) == 1)
    if (~isempty(cds.intensity_P))
        dates_scale = datenum(cds.dates{1})-datenum(cds.dates{1}(1));
        plot(dates_scale, cds.intensity_P*1e4, 'LineWidth', 1.2, 'Color', 'Red', 'LineStyle', '--')
    end
end

% Plot lambda_Q
if (ind_plots(3) == 1)
    if (~isempty(cds.x0))
        dates_scale = datenum(cds.dates{1})-datenum(cds.dates{1}(1));
        % Match dates from CDS and CDX
        [trash, pos_cdx] = is_member_sorted_c(cds.dates{1}, CDX.dates{1});
        total_lambda_Q = (cds.x0 + cds.ai .* CDX.y0(pos_cdx))* 1e4;
        plot(dates_scale, total_lambda_Q, 'LineWidth', 1.2, 'Color', 'Black', 'LineStyle', ':')
    end
end

% Plot systematic intensity
if (ind_plots(4) == 1)
    if (~isempty(cds.x0))
        dates_scale = datenum(cds.dates{1})-datenum(cds.dates{1}(1));
        % Match dates from CDS and CDX
        [trash, pos_cdx] = is_member_sorted_c(cds.dates{1}, CDX.dates{1});
        lambda_Q_syst = (cds.ai .* CDX.y0(pos_cdx))* 1e4;
        plot(dates_scale, lambda_Q_syst, 'LineWidth', 1.2, 'Color', [1 0 1], 'LineStyle', '-.')
    end
end

% Plot idiosyncratic intensity
if (ind_plots(5) == 1)
    if (~isempty(cds.x0))
        dates_scale = datenum(cds.dates{1})-datenum(cds.dates{1}(1));
        % Match dates from CDS and CDX
        [trash, pos_cdx] = is_member_sorted_c(cds.dates{1}, CDX.dates{1});
        lambda_Q_idio = cds.x0 .* cds.bi_P * 1e4;
        plot(dates_scale, lambda_Q_idio, 'LineWidth', 1.2, 'Color', [0 1 1], 'LineStyle', '-')
    end
end
hold off

% Calculate correlation changes between lambda_P and lambda_q
if (1)
    cor_lambda_P_Q = corr(total_lambda_Q, cds.intensity_P);
    disp(['Correlation of lambda_P and lambda_Q: ' num2str(cor_lambda_P_Q)]);
    cor_lambda_P_Q2 = corr(total_lambda_Q, cds.intensity_P, 'type', 'Spearman');
    disp(['Spearman rank correlation of lambda_P and lambda_Q: ' num2str(cor_lambda_P_Q2)]);
    cor_lambda_P_Q = corr(diff(total_lambda_Q), diff(cds.intensity_P));
    disp(['Correlation of changes in lambda_P and lambda_Q: ' num2str(cor_lambda_P_Q)]);
    cor_lambda_P_Q2 = corr(diff(total_lambda_Q), diff(cds.intensity_P), 'type', 'Spearman');
    disp(['Spearman rank correlation of changes in lambda_P and lambda_Q: ' num2str(cor_lambda_P_Q2)]);
end

ylabel('intensity')
all_data = [cds.market_price{maturity_pos}/cds.LGD; cds.intensity_P*1e4; total_lambda_Q];
make_nice_axis(gca, all_data);
make_nice_ticks(cds.dates{maturity_pos}, 120);
%axis([0 dates_scale(end) min(all_data) max(all_data)]);

set(gca,'ytick', [0.1 0.2 0.5 1 2 5 10 20 50 100 200 500 1000]);
%set(gca,['y','ticklabel'], num2str(ticks));

% Calculate JTD risk premium
JTD = total_lambda_Q ./ cds.intensity_P / 1e4 - 1;

