function plot_historical_tranche_spreads(CDX, start_date, end_date, used_tranches)
% --------------------------------------------------------------------------------------------------
% Visualize the historical tranche spread time series between start_date and end_date. Spreads are
% normalized to be equal to one at the beginning in order to facilitate a comparison.
% --------------------------------------------------------------------------------------------------
% CDX                   ... credit index structure
% start_date            ... start date of time window
% end_date              ... end date of time window
% used_tranches         ... indicator vector, which tranche spread time series to plot; default all
% --------------------------------------------------------------------------------------------------
% sample call: plot_historical_tranche_spreads(CDX_NA_IG2)
% --------------------------------------------------------------------------------------------------

% By default, plot time series from September 2003 to January 2007
if (nargin <= 1)
    start_date = datenum('09/01/2003');
    end_date = datenum('01/7/2007');
end
start_date = datenum(start_date);
end_date = datenum(end_date);
if (nargin <= 3)
    used_tranches = [1 1 1 1 1];
end
used_tranches = logical(used_tranches);

% Extract data
dates = CDX.dates{1};
coupon = CDX.tranche_market_price_mid{1}(:,2:end);
up_front = CDX.tranche_market_upfront_mid{1}(:,1);
data = [up_front coupon];

% Remove '09/01/2006' due to missing/bad data
% bad_pos = find(CDX.dates{1} == datenum('09/01/2006'));
% dates = [dates(1:(bad_pos-1),:); dates((bad_pos+1):end,:)];
% data = [data(1:(bad_pos-1),:); data((bad_pos+1):end,:)];

% Cut down data to desired time interval
start_pos = find(dates >= start_date, 1, 'first');
end_pos = find(dates <= end_date, 1, 'last');
used_range = start_pos:end_pos;
dates = dates(used_range);
data = data(used_range,:);

% Normalize data & select desired subset
data = data ./ repmat(data(1,:), size(data,1), 1);
data = data(:,used_tranches);

% Visualize historical tranche spreads (all tranches in one plot)
color_matrix = [[0 0 1]; [1 0 0]; [0 0 0]; [1 0 1]; [0 1 1]; ];
line_styles = {'-' '--' ':' '-.' '-' '--' ':' '-.'};
markers = {'none' 'none' 'none' 'none' 'x' 'x' 'x' 'x'};
dates_scale = dates - start_date;
plot(dates_scale, data(:,1), 'LineWidth', 1.2, 'Color', color_matrix(1,:), 'LineStyle', line_styles{1}, 'Marker', markers{1})
hold all
for j=2:size(data,2)
    plot(dates_scale, data(:,j), 'LineWidth', 1.2, 'Color', color_matrix(j,:), 'LineStyle', line_styles{j}, 'Marker', markers{j})
end
hold off

% Add labels for axes and legend
ylabel('spread')
legend_text = {'Equity', 'Junior Mezzanine', 'Mezzanine', 'Senior', 'Super Senior'};
legend(legend_text(used_tranches));
legend('Location', 'SouthEast')
%axis([0 end_date-start_date 0 1.4]);
if (length(dates) > 400)
    tick_spacing = 153;
    datestr_type = 12;
elseif (length(dates) > 100)
    tick_spacing = 30;
    datestr_type = 12;
else
    tick_spacing = 10;
    datestr_type = 'mm/dd/yy';
end
make_nice_ticks(dates, tick_spacing, dates_scale(1), datestr_type);
make_nice_axis(gca, reshape(data, numel(data), 1));
%plot_rollover_dates(start_date);
