function table2(CDX, start_date_num, end_date_num)
% --------------------------------------------------------------------------------------------------
% Creat a summary table ratings for tranche spreads, ...
% --------------------------------------------------------------------------------------------------
% CDX               ... credit index structure which this CDX is part of (because need y0 & dynamics of common factor)
% start_date_num    ... optional datenum of start date, default: use all dates
% end_date_num      ... optional datenum of end date, default: use all dates
% --------------------------------------------------------------------------------------------------
% sample call: table2(CDX_NA_IG2)
% --------------------------------------------------------------------------------------------------

% Set default parameter values
if (nargin <= 1)
    start_date_num = CDX.dates{1}(1);
    end_date_num = CDX.dates{1}(end);
end

% Allocate memory
tranches = {'Equity' 'Junior Mezzanine' 'Mezzanine' 'Senior' 'Super Senior'};
cut_offs = {'0% - 3%' '3% - 7%' '7% - 10%' '10% - 15%' '15% - 30%'};
spread_avg = zeros(size(tranches));
spread_mean = zeros(size(tranches));
spread_median = zeros(size(tranches));
spread_std = zeros(size(tranches));
spread_min = zeros(size(tranches));
spread_5_percentile = zeros(size(tranches));
spread_95_percentile = zeros(size(tranches));
spread_max = zeros(size(tranches));

% Determine relevant date range
start_pos = find(CDX.dates{1} >= start_date_num, 1, 'first');
end_pos = find(CDX.dates{1} <= end_date_num, 1, 'last');
used_range = start_pos:end_pos;

% Calculate statistics of spread for each tranche
for i=1:length(tranches)
    % Extract price
    if (i==1)
        spread = CDX.tranche_market_upfront_mid{1}(:,i) * 100;
    else
        spread = CDX.tranche_market_price_mid{1}(:,i);
    end
    
    % Filter out bad data: 01-Sep-2006 (equity), 27-Sept-2006 (all tranche), 02-Sept-2005 (SupSen)
    bad_pos = find(CDX.dates{1} == datenum('09/27/2006'));
    spread = [spread(1:(bad_pos-1)); spread((bad_pos+1):end)];
    if (i==1)
        bad_pos = find(CDX.dates{1} == datenum('09/01/2006'));
        spread = [spread(1:(bad_pos-1)); spread((bad_pos+1):end)];
    end
    if (i==5)
        bad_pos = find(CDX.dates{1} == datenum('09/06/2005'));
        spread = [spread(1:(bad_pos-1)); spread((bad_pos+1):end)];
    end
    
    % Calculate statistics
    spread_avg(i) = mean(spread);
    spread_changes = diff(spread) ./ spread(1:end-1);
    spread_mean(i) = mean(spread_changes);
    spread_median(i) = median(spread_changes);    
    spread_std(i) = std(spread_changes);
    spread_min(i) = min(spread_changes);
    spread_5_percentile(i) = prctile(spread_changes, 5);
    spread_95_percentile(i) = prctile(spread_changes, 95);
    spread_max(i) = max(spread_changes);
end

% Put everything together into a table and write into Excel file
xlswrite('c:/thesis/main paper/implementation/tables/table2.xls', {'Tranche' '% of Credit Losses' 'Average Spread' 'Mean(delta_S/S)' ...
                        'Median(delta_S/S)' 'Std(delta_S/S)' 'Min(delta_S/S)' '5%-tile(delta_S/S)' '95%-tile(delta_S/S)' 'Max(delta_S/S)'}, 1, 'A1');
xlswrite('c:/thesis/main paper/implementation/tables/table2.xls', tranches', 1, 'A2');
xlswrite('c:/thesis/main paper/implementation/tables/table2.xls', cut_offs', 1, 'B2');
xlswrite('c:/thesis/main paper/implementation/tables/table2.xls', round(spread_avg'*10)/10, 1, 'C2');
xlswrite('c:/thesis/main paper/implementation/tables/table2.xls', round(spread_mean'*1e3)/1000, 1, 'D2');
xlswrite('c:/thesis/main paper/implementation/tables/table2.xls', round(spread_median'*1e3)/1000, 1, 'E2');
xlswrite('c:/thesis/main paper/implementation/tables/table2.xls', round(spread_std'*1e3)/1000, 1, 'F2');
xlswrite('c:/thesis/main paper/implementation/tables/table2.xls', round(spread_min'*1e3)/1000, 1, 'G2');
xlswrite('c:/thesis/main paper/implementation/tables/table2.xls', round(spread_5_percentile'*1e3)/1000, 1, 'H2');
xlswrite('c:/thesis/main paper/implementation/tables/table2.xls', round(spread_95_percentile'*1e3)/1000, 1, 'I2');
xlswrite('c:/thesis/main paper/implementation/tables/table2.xls', round(spread_max'*1e3)/1000, 1, 'J2');

