function [err_real, err_imag] = transform_comparison(mu_imag, sigma_imag, N)
% --------------------------------------------------------------------------------------------------
% Compare two different implementations for calculating the Fourier transform of in integrated basic
% affine jump diffusion by randomly choosing complex numbers at which to calculate the transform.
% --------------------------------------------------------------------------------------------------
% Method 1: Solve ODEs from Appendix in Mortensen (2006)
% Method 2: Plug in complex values into formula from Appendix A in "Credit Risk" by Duffie and
%           Singleton (2003), even though their derivation is for real numbers only
% --------------------------------------------------------------------------------------------------
% mu_imag       ... mean for drawing imaginary part
% sigma_imag    ... stanford deviation on normal for drawing imaginary part
% N             ... number of points that should be drawn
% --------------------------------------------------------------------------------------------------
% sample call: [err_real, err_imag] = transform_comparison(0, 20, 20)
% --------------------------------------------------------------------------------------------------

% Draw random points
u_imag = mu_imag + randn(N, 1) * sigma_imag;
transform1 = zeros(N, 1);
transform2 = zeros(N, 1);

% Set typical AJD parameters
x0 = 0.002;
k = 0.25;
theta = 0.07;
sigma = 0.05;
l = 0.01;
mu = 0.15;
T = 5;
ajd_dynamics = struct('k', k,'theta', theta,'sigma', sigma, 'L', l,'mu', mu);

% Calculate differences in the two methods for randomly draw points
for j=1:N
    % ODE based method
    tmp = transform_integrated_ajd(u_imag(j), x0, ajd_dynamics, 0:(1/250):T);
    transform1(j) = tmp(end);
    
    % Explicit method
    [a,b] = transform2_integrated_ajd_c(0, u_imag(j), x0, k, theta, sigma, l, mu, T);
    transform2(j) = a + i*b;
end

% Calculate relative errors
err_real = real(transform2 - transform1) ./ real(transform1);
err_imag = imag(transform2 - transform1) ./ imag(transform2);
disp(['Average absolute relative error of real part: ' num2str(mean(abs(err_real)))]);
disp(['Average absolute relative error of imaginary part: ' num2str(mean(abs(err_imag)))]);

